'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Input } from '@/components/ui/input'
import { 
  Plus, 
  Edit, 
  Trash2, 
  Search, 
  Filter, 
  Eye, 
  Heart, 
  Share2,
  Calendar,
  User,
  ArrowLeft,
  LogOut,
  BarChart3,
  FileText,
  MessageSquare,
  Users as UsersIcon,
  Settings,
  Calendar as CalendarIcon
} from 'lucide-react'

export default function AdminArticles() {
  const router = useRouter()
  const [adminData, setAdminData] = useState<any>(null)
  const [articles, setArticles] = useState([])
  const [searchTerm, setSearchTerm] = useState('')
  const [selectedCategory, setSelectedCategory] = useState('all')

  useEffect(() => {
    // Get admin session from layout (already authenticated)
    const session = localStorage.getItem('admin_session')
    if (session) {
      try {
        const parsedSession = JSON.parse(session)
        setAdminData(parsedSession)
      } catch (error) {
        console.error('Error parsing admin session:', error)
      }
    }

    // Sample articles data
    setArticles([
      {
        id: 1,
        title: "Program Bantuan UMKM Sukses Diluncurkan di Jakarta",
        excerpt: "Program bantuan modal usaha untuk pelaku UMKM telah resmi diluncurkan dan mendapat sambutan positif dari masyarakat.",
        category: "Ekonomi",
        author: "Tim Media",
        date: "2024-01-15",
        status: "published",
        views: 1250,
        likes: 89,
        featured: true
      },
      {
        id: 2,
        title: "Kunjungan ke Pusat Kesehatan Masyarakat di Bekasi",
        excerpt: "Meninjau langsung kondisi fasilitas kesehatan dan berdialog dengan tenaga medis serta pasien.",
        category: "Kesehatan",
        author: "Tim Media",
        date: "2024-01-14",
        status: "published",
        views: 890,
        likes: 67,
        featured: false
      },
      {
        id: 3,
        title: "Sosialisasi Program Pendidikan Gratis di Tangerang",
        excerpt: "Mengenalkan program pendidikan gratis kepada masyarakat dan menjawab berbagai pertanyaan.",
        category: "Pendidikan",
        author: "Tim Media",
        date: "2024-01-13",
        status: "draft",
        views: 0,
        likes: 0,
        featured: false
      }
    ])
  }, [])

  const handleDelete = (id: number) => {
    if (confirm('Apakah Anda yakin ingin menghapus artikel ini?')) {
      setArticles(articles.filter((article: any) => article.id !== id))
    }
  }

  const filteredArticles = articles.filter((article: any) => {
    const matchesSearch = article.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         article.excerpt.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesCategory = selectedCategory === 'all' || article.category === selectedCategory
    return matchesSearch && matchesCategory
  })

  const categories = ['all', 'Ekonomi', 'Kesehatan', 'Pendidikan', 'Infrastruktur', 'Sosial']

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'published': return 'bg-green-100 text-green-800'
      case 'draft': return 'bg-yellow-100 text-yellow-800'
      case 'archived': return 'bg-gray-100 text-gray-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const handleLogout = () => {
    localStorage.removeItem('admin_session')
    document.cookie = 'auth-token=; path=/; expires=Thu, 01 Jan 1970 00:00:01 GMT'
    router.push('/admin/login')
  }

  if (!adminData) {
    return null
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-white border-b border-slate-200">
        <div className="px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <div className="h-8 w-8 rounded-full bg-primary flex items-center justify-center mr-3">
                <FileText className="h-4 w-4 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-semibold text-slate-900">Manajemen Artikel</h1>
                <p className="text-xs text-slate-500">Kelola semua konten berita</p>
              </div>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-right">
                <p className="text-sm font-medium text-slate-900">{adminData?.name}</p>
                <p className="text-xs text-slate-500">Online</p>
              </div>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Artikel Baru
              </Button>
              <Button variant="outline" size="sm" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar */}
        <aside className="w-64 bg-white border-r border-slate-200 min-h-screen">
          <nav className="p-4 space-y-2">
            <Link href="/admin/dashboard" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <BarChart3 className="h-5 w-5" />
              <span>Dashboard</span>
            </Link>
            <Link href="/admin/articles" className="flex items-center space-x-3 px-3 py-2 rounded-lg bg-primary text-primary-foreground">
              <FileText className="h-5 w-5" />
              <span>Artikel</span>
            </Link>
            <Link href="/admin/reports" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <MessageSquare className="h-5 w-5" />
              <span>Laporan</span>
            </Link>
            <Link href="/admin/users" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <UsersIcon className="h-5 w-5" />
              <span>Pengguna</span>
            </Link>
            <Link href="/admin/calendar" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Calendar className="h-5 w-5" />
              <span>Agenda</span>
            </Link>
            <Link href="/admin/settings" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Settings className="h-5 w-5" />
              <span>Pengaturan</span>
            </Link>
          </nav>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-6">
          {/* Filters */}
          <Card className="mb-6">
            <CardContent className="p-4">
              <div className="flex flex-col lg:flex-row gap-4">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-slate-400" />
                  <Input
                    placeholder="Cari artikel..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
                
                <select
                  value={selectedCategory}
                  onChange={(e) => setSelectedCategory(e.target.value)}
                  className="px-3 py-2 border rounded-lg bg-white"
                >
                  {categories.map(category => (
                    <option key={category} value={category}>
                      {category === 'all' ? 'Semua Kategori' : category}
                    </option>
                  ))}
                </select>

                <Button variant="outline">
                  <Filter className="h-4 w-4 mr-2" />
                  Filter
                </Button>
              </div>
            </CardContent>
          </Card>

          {/* Articles List */}
          <div className="space-y-4">
            {filteredArticles.map((article: any) => (
              <Card key={article.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-6">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center space-x-3 mb-2">
                        <h3 className="text-lg font-semibold">{article.title}</h3>
                        {article.featured && (
                          <Badge variant="secondary">Utama</Badge>
                        )}
                        <Badge className={getStatusColor(article.status)}>
                          {article.status}
                        </Badge>
                      </div>
                      
                      <p className="text-slate-600 mb-3 line-clamp-2">
                        {article.excerpt}
                      </p>
                      
                      <div className="flex items-center space-x-4 text-sm text-slate-500">
                        <span className="flex items-center">
                          <User className="h-4 w-4 mr-1" />
                          {article.author}
                        </span>
                        <span className="flex items-center">
                          <Calendar className="h-4 w-4 mr-1" />
                          {article.date}
                        </span>
                        <span className="flex items-center">
                          <Eye className="h-4 w-4 mr-1" />
                          {article.views}
                        </span>
                        <span className="flex items-center">
                          <Heart className="h-4 w-4 mr-1" />
                          {article.likes}
                        </span>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-2 ml-4">
                      <Button variant="outline" size="sm">
                        <Edit className="h-4 w-4" />
                      </Button>
                      <Button 
                        variant="outline" 
                        size="sm"
                        onClick={() => handleDelete(article.id)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {filteredArticles.length === 0 && (
            <Card>
              <CardContent className="p-12 text-center">
                <p className="text-slate-500">Tidak ada artikel yang ditemukan.</p>
                <Button className="mt-4">
                  <Plus className="h-4 w-4 mr-2" />
                  Buat Artikel Baru
                </Button>
              </CardContent>
            </Card>
          )}
        </main>
      </div>
    </div>
  )
}