'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { 
  Calendar,
  Clock,
  MapPin,
  User,
  Plus,
  Edit,
  Trash2,
  ArrowLeft,
  Eye,
  LogOut,
  BarChart3,
  FileText,
  MessageSquare,
  Users as UsersIcon,
  Settings,
  Calendar as CalendarIcon
} from 'lucide-react'
import { format } from 'date-fns'
import { id } from 'date-fns/locale'

export default function AdminCalendar() {
  const router = useRouter()
  const [adminData, setAdminData] = useState<any>(null)

  useEffect(() => {
    // Get admin session from layout (already authenticated)
    const session = localStorage.getItem('admin_session')
    if (session) {
      try {
        const parsedSession = JSON.parse(session)
        setAdminData(parsedSession)
      } catch (error) {
        console.error('Error parsing admin session:', error)
      }
    }
  }, [])

  // Sample calendar events data
  const events = [
    {
      id: 1,
      title: "Rapat Evaluasi Program Kesehatan",
      date: "2024-01-20",
      time: "09:00",
      location: "Kantor Walikota Jakarta Selatan",
      type: "meeting",
      attendees: 25
    },
    {
      id: 2,
      title: "Kunjungan Lapangan ke Pusat Kesehatan",
      date: "2024-01-22",
      time: "14:00",
      location: "Puskesmas Kebayoran Baru",
      type: "visit",
      attendees: 8
    },
    {
      id: 3,
      title: "Sosialisasi Program Pendidikan Gratis",
      date: "2024-01-25",
      time: "10:00",
      location: "Gedung Serbaguna Tangerang",
      type: "event",
      attendees: 150
    }
  ]

  const getTypeColor = (type: string) => {
    switch (type) {
      case 'meeting': return 'bg-blue-100 text-blue-800'
      case 'visit': return 'bg-green-100 text-green-800'
      case 'event': return 'bg-purple-100 text-purple-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const handleLogout = () => {
    localStorage.removeItem('admin_session')
    document.cookie = 'auth-token=; path=/; expires=Thu, 01 Jan 1970 00:00:01 GMT'
    router.push('/admin/login')
  }

  if (!adminData) {
    return null
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-white border-b border-slate-200">
        <div className="px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <div className="h-8 w-8 rounded-full bg-primary flex items-center justify-center mr-3">
                <CalendarIcon className="h-4 w-4 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-semibold text-slate-900">Kalender Agenda</h1>
                <p className="text-xs text-slate-500">Kelola jadwal kegiatan</p>
              </div>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-right">
                <p className="text-sm font-medium text-slate-900">{adminData?.name}</p>
                <p className="text-xs text-slate-500">Online</p>
              </div>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Tambah Agenda
              </Button>
              <Button variant="outline" size="sm" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar */}
        <aside className="w-64 bg-white border-r border-slate-200 min-h-screen">
          <nav className="p-4 space-y-2">
            <Link href="/admin/dashboard" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <BarChart3 className="h-5 w-5" />
              <span>Dashboard</span>
            </Link>
            <Link href="/admin/articles" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <FileText className="h-5 w-5" />
              <span>Artikel</span>
            </Link>
            <Link href="/admin/reports" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <MessageSquare className="h-5 w-5" />
              <span>Laporan</span>
            </Link>
            <Link href="/admin/users" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <UsersIcon className="h-5 w-5" />
              <span>Pengguna</span>
            </Link>
            <Link href="/admin/calendar" className="flex items-center space-x-3 px-3 py-2 rounded-lg bg-primary text-primary-foreground">
              <CalendarIcon className="h-5 w-5" />
              <span>Agenda</span>
            </Link>
            <Link href="/admin/settings" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Settings className="h-5 w-5" />
              <span>Pengaturan</span>
            </Link>
          </nav>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-6">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Events List */}
            <div className="lg:col-span-2 space-y-4">
              <Card>
                <CardHeader>
                  <CardTitle>Daftar Agenda</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {events.map((event: any) => (
                      <div key={event.id} className="border rounded-lg p-4 hover:shadow-md transition-shadow">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <div className="flex items-center space-x-3 mb-2">
                              <h3 className="font-semibold text-lg">{event.title}</h3>
                              <Badge className={getTypeColor(event.type)}>
                                {event.type.charAt(0).toUpperCase() + event.type.slice(1)}
                              </Badge>
                            </div>
                            
                            <div className="space-y-2">
                              <div className="flex items-center text-sm text-slate-600">
                                <Calendar className="h-4 w-4 mr-2" />
                                {new Date(event.date).toLocaleDateString('id-ID', { 
                                  weekday: 'long', 
                                  year: 'numeric', 
                                  month: 'long', 
                                  day: 'numeric' 
                                })}
                              </div>
                              
                              <div className="flex items-center text-sm text-slate-600">
                                <Clock className="h-4 w-4 mr-2" />
                                {event.time}
                              </div>
                              
                              <div className="flex items-center text-sm text-slate-600">
                                <MapPin className="h-4 w-4 mr-2" />
                                {event.location}
                              </div>
                              
                              <div className="flex items-center text-sm text-slate-600">
                                <User className="h-4 w-4 mr-2" />
                                {event.attendees} peserta
                              </div>
                            </div>
                          </div>
                          
                          <div className="flex items-center space-x-2 ml-4">
                            <Button variant="outline" size="sm">
                              <Eye className="h-4 w-4" />
                            </Button>
                            <Button variant="outline" size="sm">
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button variant="outline" size="sm">
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Calendar Preview */}
            <div>
              <Card>
                <CardHeader>
                  <CardTitle>Pratinjau Kalender</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="bg-slate-50 p-4 rounded-lg">
                    <div className="text-center mb-4">
                      <h3 className="font-semibold">Januari 2024</h3>
                    </div>
                    <div className="grid grid-cols-7 gap-1 text-xs text-center mb-1">
                      {['Min', 'Sen', 'Sel', 'Rab', 'Kam', 'Jum', 'Sab'].map((day, i) => (
                        <div key={i} className="font-medium py-1 text-slate-500">{day}</div>
                      ))}
                    </div>
                    <div className="grid grid-cols-7 gap-1">
                      {Array.from({ length: 31 }, (_, i) => {
                        const day = i + 1;
                        const hasEvent = events.some(event => new Date(event.date).getDate() === day);
                        
                        return (
                          <div 
                            key={day} 
                            className={`h-8 flex items-center justify-center text-sm rounded ${
                              hasEvent ? 'bg-blue-100 text-blue-800 font-medium' : 'text-slate-600'
                            }`}
                          >
                            {day}
                          </div>
                        );
                      })}
                    </div>
                  </div>
                  
                  <div className="mt-6 space-y-3">
                    <h4 className="font-medium">Legenda:</h4>
                    <div className="flex items-center space-x-2">
                      <div className="h-3 w-3 bg-blue-100 rounded"></div>
                      <span className="text-sm">Meeting</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <div className="h-3 w-3 bg-green-100 rounded"></div>
                      <span className="text-sm">Kunjungan Lapangan</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <div className="h-3 w-3 bg-purple-100 rounded"></div>
                      <span className="text-sm">Event Publik</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </main>
      </div>
    </div>
  )
}