'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Progress } from '@/components/ui/progress'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { 
  BarChart3, 
  Users, 
  FileText, 
  MessageSquare, 
  Calendar,
  TrendingUp,
  TrendingDown,
  Eye,
  Heart,
  Share2,
  Clock,
  CheckCircle,
  AlertCircle,
  XCircle,
  LogOut,
  Settings,
  Plus,
  Edit,
  Trash2,
  Search,
  Filter
} from 'lucide-react'

export default function AdminDashboard() {
  const router = useRouter()
  const [adminData, setAdminData] = useState<any>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // Get admin session from layout (already authenticated)
    const session = localStorage.getItem('admin_session')
    if (session) {
      try {
        const parsedSession = JSON.parse(session)
        setAdminData(parsedSession)
      } catch (error) {
        console.error('Error parsing admin session:', error)
      }
    }
    setIsLoading(false)
  }, [])

  const handleLogout = () => {
    localStorage.removeItem('admin_session')
    document.cookie = 'auth-token=; path=/; expires=Thu, 01 Jan 1970 00:00:01 GMT'
    router.push('/admin/login')
  }

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto mb-4"></div>
          <p>Loading...</p>
        </div>
      </div>
    )
  }

  if (!adminData) {
    return null
  }

  // Sample data untuk dashboard
  const stats = {
    totalVisitors: 45234,
    totalArticles: 156,
    totalReports: 89,
    totalMessages: 234,
    monthlyGrowth: 12.5,
    engagementRate: 68.9
  }

  const recentReports = [
    {
      id: 1,
      title: "Jalan Rusak di Jakarta Selatan",
      category: "Infrastruktur",
      status: "PROCESSING",
      date: "2024-01-15",
      author: "Budi Santoso"
    },
    {
      id: 2,
      title: "Bantuan Sosial Belum Tersalur",
      category: "Sosial",
      status: "RECEIVED",
      date: "2024-01-14",
      author: "Siti Nurhaliza"
    },
    {
      id: 3,
      title: "Sampah Menumpuk di TPA",
      category: "Lingkungan",
      status: "COMPLETED",
      date: "2024-01-13",
      author: "Ahmad Fauzi"
    }
  ]

  const recentArticles = [
    {
      id: 1,
      title: "Program Bantuan UMKM Sukses Diluncurkan",
      category: "Ekonomi",
      views: 1250,
      likes: 89,
      date: "2024-01-15"
    },
    {
      id: 2,
      title: "Kunjungan ke Pusat Kesehatan Masyarakat",
      category: "Kesehatan",
      views: 890,
      likes: 67,
      date: "2024-01-14"
    },
    {
      id: 3,
      title: "Sosialisasi Program Pendidikan Gratis",
      category: "Pendidikan",
      views: 1100,
      likes: 92,
      date: "2024-01-13"
    }
  ]

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'RECEIVED': return 'bg-blue-100 text-blue-800'
      case 'PROCESSING': return 'bg-yellow-100 text-yellow-800'
      case 'ADDRESSED': return 'bg-purple-100 text-purple-800'
      case 'COMPLETED': return 'bg-green-100 text-green-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'RECEIVED': return <AlertCircle className="h-4 w-4" />
      case 'PROCESSING': return <Clock className="h-4 w-4" />
      case 'ADDRESSED': return <Edit className="h-4 w-4" />
      case 'COMPLETED': return <CheckCircle className="h-4 w-4" />
      default: return <XCircle className="h-4 w-4" />
    }
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-white border-b border-slate-200">
        <div className="px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Link href="/admin/dashboard" className="flex items-center space-x-3">
                <div className="h-8 w-8 rounded-full bg-primary flex items-center justify-center">
                  <BarChart3 className="h-4 w-4 text-primary-foreground" />
                </div>
                <div>
                  <h1 className="text-xl font-semibold text-slate-900">Admin Dashboard</h1>
                  <p className="text-xs text-slate-500">Politisi Indonesia</p>
                </div>
              </Link>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-right">
                <p className="text-sm font-medium text-slate-900">{adminData.name}</p>
                <p className="text-xs text-slate-500">Online</p>
              </div>
              <Button variant="outline" size="sm" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar */}
        <aside className="w-64 bg-white border-r border-slate-200 min-h-screen">
          <nav className="p-4 space-y-2">
            <Link href="/admin/dashboard" className="flex items-center space-x-3 px-3 py-2 rounded-lg bg-primary text-primary-foreground">
              <BarChart3 className="h-5 w-5" />
              <span>Dashboard</span>
            </Link>
            <Link href="/admin/articles" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <FileText className="h-5 w-5" />
              <span>Artikel</span>
            </Link>
            <Link href="/admin/reports" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <MessageSquare className="h-5 w-5" />
              <span>Laporan</span>
            </Link>
            <Link href="/admin/users" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Users className="h-5 w-5" />
              <span>Pengguna</span>
            </Link>
            <Link href="/admin/calendar" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Calendar className="h-5 w-5" />
              <span>Agenda</span>
            </Link>
            <Link href="/admin/settings" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Settings className="h-5 w-5" />
              <span>Pengaturan</span>
            </Link>
          </nav>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-6">
          {/* Welcome Section */}
          <div className="mb-8">
            <h2 className="text-2xl font-bold text-slate-900 mb-2">
              Selamat datang kembali, {adminData.name}!
            </h2>
            <p className="text-slate-600">
              Ini adalah ringkasan aktivitas dan statistik terkini dari website Anda.
            </p>
          </div>

          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Pengunjung</CardTitle>
                <Users className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats.totalVisitors.toLocaleString()}</div>
                <p className="text-xs text-muted-foreground">
                  <span className="text-green-600 flex items-center">
                    <TrendingUp className="h-3 w-3 mr-1" />
                    +{stats.monthlyGrowth}% dari bulan lalu
                  </span>
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Total Artikel</CardTitle>
                <FileText className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats.totalArticles}</div>
                <p className="text-xs text-muted-foreground">
                  <span className="text-blue-600">12 artikel bulan ini</span>
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Laporan Masuk</CardTitle>
                <MessageSquare className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats.totalReports}</div>
                <p className="text-xs text-muted-foreground">
                  <span className="text-yellow-600">23 menunggu respons</span>
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Engagement Rate</CardTitle>
                <TrendingUp className="h-4 w-4 text-muted-foreground" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stats.engagementRate}%</div>
                <p className="text-xs text-muted-foreground">
                  <span className="text-green-600 flex items-center">
                    <TrendingUp className="h-3 w-3 mr-1" />
                    +5.2% dari minggu lalu
                  </span>
                </p>
              </CardContent>
            </Card>
          </div>

          {/* Content Tabs */}
          <Tabs defaultValue="overview" className="space-y-6">
            <TabsList>
              <TabsTrigger value="overview">Ringkasan</TabsTrigger>
              <TabsTrigger value="reports">Laporan Terbaru</TabsTrigger>
              <TabsTrigger value="articles">Artikel Populer</TabsTrigger>
              <TabsTrigger value="analytics">Analitik</TabsTrigger>
            </TabsList>

            <TabsContent value="overview" className="space-y-6">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* Recent Reports */}
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                      Laporan Terbaru
                      <Link href="/admin/reports">
                        <Button variant="outline" size="sm">
                          Lihat Semua
                        </Button>
                      </Link>
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {recentReports.map((report) => (
                        <div key={report.id} className="flex items-center justify-between p-3 border rounded-lg">
                          <div className="flex-1">
                            <h4 className="font-medium text-sm">{report.title}</h4>
                            <p className="text-xs text-slate-500">
                              {report.category} • {report.author} • {report.date}
                            </p>
                          </div>
                          <Badge className={`flex items-center space-x-1 ${getStatusColor(report.status)}`}>
                            {getStatusIcon(report.status)}
                            <span className="text-xs">{report.status}</span>
                          </Badge>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>

                {/* Recent Articles */}
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                      Artikel Populer
                      <Link href="/admin/articles">
                        <Button variant="outline" size="sm">
                          Kelola
                        </Button>
                      </Link>
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {recentArticles.map((article) => (
                        <div key={article.id} className="flex items-center justify-between p-3 border rounded-lg">
                          <div className="flex-1">
                            <h4 className="font-medium text-sm">{article.title}</h4>
                            <p className="text-xs text-slate-500">
                              {article.category} • {article.date}
                            </p>
                          </div>
                          <div className="flex items-center space-x-2 text-xs text-slate-500">
                            <span className="flex items-center">
                              <Eye className="h-3 w-3 mr-1" />
                              {article.views}
                            </span>
                            <span className="flex items-center">
                              <Heart className="h-3 w-3 mr-1" />
                              {article.likes}
                            </span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              </div>
            </TabsContent>

            <TabsContent value="reports">
              <Card>
                <CardHeader>
                  <CardTitle>Semua Laporan</CardTitle>
                  <CardDescription>
                    Kelola dan respons laporan dari masyarakat
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center space-x-4 mb-6">
                    <div className="relative flex-1">
                      <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-slate-400" />
                      <input
                        type="text"
                        placeholder="Cari laporan..."
                        className="w-full pl-10 pr-4 py-2 border rounded-lg"
                      />
                    </div>
                    <Button variant="outline">
                      <Filter className="h-4 w-4 mr-2" />
                      Filter
                    </Button>
                    <Button>
                      <Plus className="h-4 w-4 mr-2" />
                      Laporan Baru
                    </Button>
                  </div>
                  
                  <div className="space-y-4">
                    {recentReports.map((report) => (
                      <div key={report.id} className="border rounded-lg p-4">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h3 className="font-semibold">{report.title}</h3>
                            <p className="text-sm text-slate-600 mt-1">
                              Oleh: {report.author} • {report.date}
                            </p>
                            <Badge className="mt-2">{report.category}</Badge>
                          </div>
                          <div className="flex items-center space-x-2">
                            <Badge className={`flex items-center space-x-1 ${getStatusColor(report.status)}`}>
                              {getStatusIcon(report.status)}
                              <span>{report.status}</span>
                            </Badge>
                            <Button variant="outline" size="sm">
                              <Edit className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="articles">
              <Card>
                <CardHeader>
                  <CardTitle>Manajemen Artikel</CardTitle>
                  <CardDescription>
                    Kelola semua artikel dan konten website
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center space-x-4 mb-6">
                    <Button>
                      <Plus className="h-4 w-4 mr-2" />
                      Artikel Baru
                    </Button>
                  </div>
                  
                  <div className="space-y-4">
                    {recentArticles.map((article) => (
                      <div key={article.id} className="border rounded-lg p-4">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h3 className="font-semibold">{article.title}</h3>
                            <p className="text-sm text-slate-600 mt-1">
                              {article.category} • {article.date}
                            </p>
                            <div className="flex items-center space-x-4 mt-2 text-sm text-slate-500">
                              <span className="flex items-center">
                                <Eye className="h-4 w-4 mr-1" />
                                {article.views} views
                              </span>
                              <span className="flex items-center">
                                <Heart className="h-4 w-4 mr-1" />
                                {article.likes} likes
                              </span>
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <Button variant="outline" size="sm">
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button variant="outline" size="sm">
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="analytics">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <Card>
                  <CardHeader>
                    <CardTitle>Trafik Pengunjung</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      <div className="flex justify-between items-center">
                        <span className="text-sm">Hari ini</span>
                        <span className="font-semibold">1,234</span>
                      </div>
                      <Progress value={75} className="h-2" />
                      <div className="flex justify-between items-center">
                        <span className="text-sm">Minggu ini</span>
                        <span className="font-semibold">8,567</span>
                      </div>
                      <Progress value={60} className="h-2" />
                      <div className="flex justify-between items-center">
                        <span className="text-sm">Bulan ini</span>
                        <span className="font-semibold">45,234</span>
                      </div>
                      <Progress value={85} className="h-2" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader>
                    <CardTitle>Konten Populer</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {recentArticles.slice(0, 3).map((article, index) => (
                        <div key={article.id} className="flex items-center space-x-3">
                          <div className="w-8 h-8 rounded-full bg-primary/10 flex items-center justify-center text-sm font-medium">
                            {index + 1}
                          </div>
                          <div className="flex-1">
                            <p className="text-sm font-medium line-clamp-1">{article.title}</p>
                            <p className="text-xs text-slate-500">{article.views} views</p>
                          </div>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              </div>
            </TabsContent>
          </Tabs>
        </main>
      </div>
    </div>
  )
}