'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Input } from '@/components/ui/input'
import { 
  Search, 
  Filter, 
  Eye, 
  MessageSquare,
  Calendar,
  User,
  ArrowLeft,
  CheckCircle,
  AlertCircle,
  Clock,
  LogOut,
  BarChart3,
  FileText,
  Users as UsersIcon,
  Settings,
  Calendar as CalendarIcon
} from 'lucide-react'

export default function AdminReports() {
  const router = useRouter()
  const [adminData, setAdminData] = useState<any>(null)

  useEffect(() => {
    // Get admin session from layout (already authenticated)
    const session = localStorage.getItem('admin_session')
    if (session) {
      try {
        const parsedSession = JSON.parse(session)
        setAdminData(parsedSession)
      } catch (error) {
        console.error('Error parsing admin session:', error)
      }
    }
  }, [])

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'RECEIVED': return 'bg-blue-100 text-blue-800'
      case 'PROCESSING': return 'bg-yellow-100 text-yellow-800'
      case 'ADDRESSED': return 'bg-purple-100 text-purple-800'
      case 'COMPLETED': return 'bg-green-100 text-green-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'RECEIVED': return <AlertCircle className="h-4 w-4" />
      case 'PROCESSING': return <Clock className="h-4 w-4" />
      case 'ADDRESSED': return <CheckCircle className="h-4 w-4" />
      case 'COMPLETED': return <CheckCircle className="h-4 w-4" />
      default: return <AlertCircle className="h-4 w-4" />
    }
  }

  // Sample reports data
  const reports = [
    {
      id: 1,
      title: "Jalan Rusak di Jakarta Selatan",
      excerpt: "Laporan kerusakan jalan yang membahayakan pengguna kendaraan",
      category: "Infrastruktur",
      author: "Budi Santoso",
      date: "2024-01-15",
      status: "PROCESSING",
      responses: 3
    },
    {
      id: 2,
      title: "Bantuan Sosial Belum Tersalur",
      excerpt: "Laporan bahwa bantuan sosial belum diterima oleh warga terdampak",
      category: "Sosial",
      author: "Siti Nurhaliza",
      date: "2024-01-14",
      status: "RECEIVED",
      responses: 1
    },
    {
      id: 3,
      title: "Sampah Menumpuk di TPA",
      excerpt: "Laporan penumpukan sampah yang menimbulkan bau tidak sedap",
      category: "Lingkungan",
      author: "Ahmad Fauzi",
      date: "2024-01-13",
      status: "COMPLETED",
      responses: 5
    }
  ]

  const handleLogout = () => {
    localStorage.removeItem('admin_session')
    document.cookie = 'auth-token=; path=/; expires=Thu, 01 Jan 1970 00:00:01 GMT'
    router.push('/admin/login')
  }

  if (!adminData) {
    return null
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-white border-b border-slate-200">
        <div className="px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <div className="h-8 w-8 rounded-full bg-primary flex items-center justify-center mr-3">
                <MessageSquare className="h-4 w-4 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-semibold text-slate-900">Manajemen Laporan</h1>
                <p className="text-xs text-slate-500">Kelola laporan dari masyarakat</p>
              </div>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-right">
                <p className="text-sm font-medium text-slate-900">{adminData?.name}</p>
                <p className="text-xs text-slate-500">Online</p>
              </div>
              <Button variant="outline">
                <MessageSquare className="h-4 w-4 mr-2" />
                Import Laporan
              </Button>
              <Button variant="outline" size="sm" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar */}
        <aside className="w-64 bg-white border-r border-slate-200 min-h-screen">
          <nav className="p-4 space-y-2">
            <Link href="/admin/dashboard" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <BarChart3 className="h-5 w-5" />
              <span>Dashboard</span>
            </Link>
            <Link href="/admin/articles" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <FileText className="h-5 w-5" />
              <span>Artikel</span>
            </Link>
            <Link href="/admin/reports" className="flex items-center space-x-3 px-3 py-2 rounded-lg bg-primary text-primary-foreground">
              <MessageSquare className="h-5 w-5" />
              <span>Laporan</span>
            </Link>
            <Link href="/admin/users" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <UsersIcon className="h-5 w-5" />
              <span>Pengguna</span>
            </Link>
            <Link href="/admin/calendar" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <CalendarIcon className="h-5 w-5" />
              <span>Agenda</span>
            </Link>
            <Link href="/admin/settings" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Settings className="h-5 w-5" />
              <span>Pengaturan</span>
            </Link>
          </nav>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-6">
          {/* Filters */}
          <Card className="mb-6">
            <CardContent className="p-4">
              <div className="flex flex-col lg:flex-row gap-4">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-slate-400" />
                  <Input
                    placeholder="Cari laporan..."
                    className="pl-10"
                  />
                </div>
                
                <select className="px-3 py-2 border rounded-lg bg-white">
                  <option value="all">Semua Kategori</option>
                  <option value="infrastruktur">Infrastruktur</option>
                  <option value="sosial">Sosial</option>
                  <option value="lingkungan">Lingkungan</option>
                  <option value="ekonomi">Ekonomi</option>
                </select>

                <select className="px-3 py-2 border rounded-lg bg-white">
                  <option value="all">Semua Status</option>
                  <option value="RECEIVED">Diterima</option>
                  <option value="PROCESSING">Diproses</option>
                  <option value="ADDRESSED">Direspon</option>
                  <option value="COMPLETED">Selesai</option>
                </select>

                <Button variant="outline">
                  <Filter className="h-4 w-4 mr-2" />
                  Filter
                </Button>
              </div>
            </CardContent>
          </Card>

          {/* Reports List */}
          <div className="space-y-4">
            {reports.map((report: any) => (
              <Card key={report.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-6">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center space-x-3 mb-2">
                        <h3 className="text-lg font-semibold">{report.title}</h3>
                        <Badge className={getStatusColor(report.status)}>
                          <span className="flex items-center">
                            {getStatusIcon(report.status)}
                            <span className="ml-1">{report.status}</span>
                          </span>
                        </Badge>
                      </div>
                      
                      <p className="text-slate-600 mb-3 line-clamp-2">
                        {report.excerpt}
                      </p>
                      
                      <div className="flex items-center space-x-4 text-sm text-slate-500">
                        <span className="flex items-center">
                          <User className="h-4 w-4 mr-1" />
                          {report.author}
                        </span>
                        <span className="flex items-center">
                          <Calendar className="h-4 w-4 mr-1" />
                          {report.date}
                        </span>
                        <span className="flex items-center">
                          <MessageSquare className="h-4 w-4 mr-1" />
                          {report.responses} respon
                        </span>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-2 ml-4">
                      <Button variant="outline" size="sm">
                        <Eye className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {reports.length === 0 && (
            <Card>
              <CardContent className="p-12 text-center">
                <p className="text-slate-500">Tidak ada laporan yang ditemukan.</p>
              </CardContent>
            </Card>
          )}
        </main>
      </div>
    </div>
  )
}