'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { 
  Settings,
  User,
  Shield,
  Lock,
  Mail,
  Globe,
  Bell,
  ArrowLeft,
  LogOut,
  BarChart3,
  FileText,
  MessageSquare,
  Users as UsersIcon,
  Calendar as CalendarIcon
} from 'lucide-react'

export default function AdminSettings() {
  const router = useRouter()
  const [adminData, setAdminData] = useState<any>(null)
  const [activeTab, setActiveTab] = useState('profile')

  useEffect(() => {
    // Get admin session from layout (already authenticated)
    const session = localStorage.getItem('admin_session')
    if (session) {
      try {
        const parsedSession = JSON.parse(session)
        setAdminData(parsedSession)
      } catch (error) {
        console.error('Error parsing admin session:', error)
      }
    }
  }, [])

  const handleLogout = () => {
    localStorage.removeItem('admin_session')
    document.cookie = 'auth-token=; path=/; expires=Thu, 01 Jan 1970 00:00:01 GMT'
    router.push('/admin/login')
  }

  if (!adminData) {
    return null
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-white border-b border-slate-200">
        <div className="px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <div className="h-8 w-8 rounded-full bg-primary flex items-center justify-center mr-3">
                <Settings className="h-4 w-4 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-semibold text-slate-900">Pengaturan Admin</h1>
                <p className="text-xs text-slate-500">Atur profil dan preferensi sistem</p>
              </div>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-right">
                <p className="text-sm font-medium text-slate-900">{adminData?.name}</p>
                <p className="text-xs text-slate-500">Online</p>
              </div>
              <Button variant="outline" size="sm" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar */}
        <aside className="w-64 bg-white border-r border-slate-200 min-h-screen">
          <nav className="p-4 space-y-2">
            <Link href="/admin/dashboard" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <BarChart3 className="h-5 w-5" />
              <span>Dashboard</span>
            </Link>
            <Link href="/admin/articles" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <FileText className="h-5 w-5" />
              <span>Artikel</span>
            </Link>
            <Link href="/admin/reports" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <MessageSquare className="h-5 w-5" />
              <span>Laporan</span>
            </Link>
            <Link href="/admin/users" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <UsersIcon className="h-5 w-5" />
              <span>Pengguna</span>
            </Link>
            <Link href="/admin/calendar" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <CalendarIcon className="h-5 w-5" />
              <span>Agenda</span>
            </Link>
            <Link href="/admin/settings" className="flex items-center space-x-3 px-3 py-2 rounded-lg bg-primary text-primary-foreground">
              <Settings className="h-5 w-5" />
              <span>Pengaturan</span>
            </Link>
          </nav>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-6">
          <div className="max-w-4xl">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Settings className="h-5 w-5 mr-2" />
                  Pengaturan Sistem
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="border-b border-slate-200 mb-6">
                  <nav className="flex space-x-8">
                    <button
                      onClick={() => setActiveTab('profile')}
                      className={`py-2 px-1 border-b-2 font-medium text-sm ${
                        activeTab === 'profile'
                          ? 'border-primary text-primary'
                          : 'border-transparent text-slate-500 hover:text-slate-700 hover:border-slate-300'
                      }`}
                    >
                      Profil
                    </button>
                    <button
                      onClick={() => setActiveTab('security')}
                      className={`py-2 px-1 border-b-2 font-medium text-sm ${
                        activeTab === 'security'
                          ? 'border-primary text-primary'
                          : 'border-transparent text-slate-500 hover:text-slate-700 hover:border-slate-300'
                      }`}
                    >
                      Keamanan
                    </button>
                    <button
                      onClick={() => setActiveTab('notifications')}
                      className={`py-2 px-1 border-b-2 font-medium text-sm ${
                        activeTab === 'notifications'
                          ? 'border-primary text-primary'
                          : 'border-transparent text-slate-500 hover:text-slate-700 hover:border-slate-300'
                      }`}
                    >
                      Notifikasi
                    </button>
                    <button
                      onClick={() => setActiveTab('general')}
                      className={`py-2 px-1 border-b-2 font-medium text-sm ${
                        activeTab === 'general'
                          ? 'border-primary text-primary'
                          : 'border-transparent text-slate-500 hover:text-slate-700 hover:border-slate-300'
                      }`}
                    >
                      Umum
                    </button>
                  </nav>
                </div>

                {/* Profile Tab */}
                {activeTab === 'profile' && (
                  <div className="space-y-6">
                    <div>
                      <h3 className="text-lg font-medium">Profil Admin</h3>
                      <p className="text-sm text-slate-600">Kelola informasi profil Anda</p>
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-2">
                        <Label htmlFor="name">Nama Lengkap</Label>
                        <Input id="name" defaultValue={adminData?.name || "Admin"} />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="email">Email</Label>
                        <Input id="email" type="email" defaultValue="admin@example.com" />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="role">Role</Label>
                        <Input id="role" defaultValue={adminData?.role || "Super Admin"} readOnly />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="phone">Nomor Telepon</Label>
                        <Input id="phone" defaultValue="+62 812-3456-7890" />
                      </div>
                    </div>
                    
                    <div className="flex justify-end">
                      <Button>
                        <User className="h-4 w-4 mr-2" />
                        Simpan Perubahan
                      </Button>
                    </div>
                  </div>
                )}

                {/* Security Tab */}
                {activeTab === 'security' && (
                  <div className="space-y-6">
                    <div>
                      <h3 className="text-lg font-medium">Keamanan</h3>
                      <p className="text-sm text-slate-600">Atur keamanan akun Anda</p>
                    </div>
                    
                    <div className="space-y-4">
                      <div className="space-y-2">
                        <Label htmlFor="current-password">Password Saat Ini</Label>
                        <Input id="current-password" type="password" />
                      </div>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div className="space-y-2">
                          <Label htmlFor="new-password">Password Baru</Label>
                          <Input id="new-password" type="password" />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="confirm-password">Konfirmasi Password</Label>
                          <Input id="confirm-password" type="password" />
                        </div>
                      </div>
                    </div>
                    
                    <div className="flex justify-end">
                      <Button>
                        <Lock className="h-4 w-4 mr-2" />
                        Ganti Password
                      </Button>
                    </div>
                  </div>
                )}

                {/* Notifications Tab */}
                {activeTab === 'notifications' && (
                  <div className="space-y-6">
                    <div>
                      <h3 className="text-lg font-medium">Notifikasi</h3>
                      <p className="text-sm text-slate-600">Atur preferensi notifikasi Anda</p>
                    </div>
                    
                    <div className="space-y-4">
                      <div className="flex items-center justify-between">
                        <div>
                          <Label className="font-medium">Email Notifikasi</Label>
                          <p className="text-sm text-slate-500">Terima notifikasi penting melalui email</p>
                        </div>
                        <Button variant="outline" size="sm">Aktif</Button>
                      </div>
                      
                      <div className="flex items-center justify-between">
                        <div>
                          <Label className="font-medium">Notifikasi Website</Label>
                          <p className="text-sm text-slate-500">Notifikasi dalam dashboard admin</p>
                        </div>
                        <Button variant="outline" size="sm">Aktif</Button>
                      </div>
                      
                      <div className="flex items-center justify-between">
                        <div>
                          <Label className="font-medium">Laporan Baru</Label>
                          <p className="text-sm text-slate-500">Dapatkan notifikasi saat ada laporan baru</p>
                        </div>
                        <Button variant="outline" size="sm">Aktif</Button>
                      </div>
                      
                      <div className="flex items-center justify-between">
                        <div>
                          <Label className="font-medium">Komentar Baru</Label>
                          <p className="text-sm text-slate-500">Notifikasi saat ada komentar baru</p>
                        </div>
                        <Button variant="outline" size="sm">Nonaktif</Button>
                      </div>
                    </div>
                  </div>
                )}

                {/* General Tab */}
                {activeTab === 'general' && (
                  <div className="space-y-6">
                    <div>
                      <h3 className="text-lg font-medium">Pengaturan Umum</h3>
                      <p className="text-sm text-slate-600">Pengaturan sistem umum</p>
                    </div>
                    
                    <div className="space-y-4">
                      <div className="space-y-2">
                        <Label htmlFor="site-name">Nama Website</Label>
                        <Input id="site-name" defaultValue="Politisi Indonesia" />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="site-email">Email Kontak</Label>
                        <Input id="site-email" type="email" defaultValue="contact@politisi.id" />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="timezone">Zona Waktu</Label>
                        <select id="timezone" className="w-full px-3 py-2 border rounded-lg bg-white">
                          <option>Asia/Jakarta (WIB)</option>
                          <option>Asia/Makassar (WITA)</option>
                          <option>Asia/Jayapura (WIT)</option>
                        </select>
                      </div>
                      
                      <div className="pt-4">
                        <h4 className="font-medium mb-2">Batasan</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div className="space-y-2">
                            <Label htmlFor="article-limit">Batas Artikel Harian</Label>
                            <Input id="article-limit" type="number" defaultValue="10" />
                          </div>
                          <div className="space-y-2">
                            <Label htmlFor="report-limit">Batas Laporan Harian</Label>
                            <Input id="report-limit" type="number" defaultValue="50" />
                          </div>
                        </div>
                      </div>
                    </div>
                    
                    <div className="flex justify-end space-x-2">
                      <Button variant="outline">Reset ke Default</Button>
                      <Button>
                        <Globe className="h-4 w-4 mr-2" />
                        Simpan Pengaturan
                      </Button>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </main>
      </div>
    </div>
  )
}