'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Input } from '@/components/ui/input'
import { 
  Search, 
  Filter, 
  User,
  Mail,
  Phone,
  Calendar,
  ArrowLeft,
  Shield,
  LogOut,
  BarChart3,
  FileText,
  MessageSquare,
  Users as UsersIcon,
  Settings,
  Calendar as CalendarIcon
} from 'lucide-react'

export default function AdminUsers() {
  const router = useRouter()
  const [adminData, setAdminData] = useState<any>(null)

  useEffect(() => {
    // Get admin session from layout (already authenticated)
    const session = localStorage.getItem('admin_session')
    if (session) {
      try {
        const parsedSession = JSON.parse(session)
        setAdminData(parsedSession)
      } catch (error) {
        console.error('Error parsing admin session:', error)
      }
    }
  }, [])

  // Sample users data
  const users = [
    {
      id: 1,
      name: "Budi Santoso",
      email: "budi@example.com",
      role: "citizen",
      joinedDate: "2024-01-15",
      reportsCount: 12,
      status: "active"
    },
    {
      id: 2,
      name: "Siti Nurhaliza",
      email: "siti@example.com",
      role: "citizen",
      joinedDate: "2024-01-14",
      reportsCount: 5,
      status: "active"
    },
    {
      id: 3,
      name: "Ahmad Fauzi",
      email: "ahmad@example.com",
      role: "citizen",
      joinedDate: "2024-01-13",
      reportsCount: 8,
      status: "active"
    }
  ]

  const getRoleColor = (role: string) => {
    switch (role) {
      case 'admin': return 'bg-blue-100 text-blue-800'
      case 'moderator': return 'bg-purple-100 text-purple-800'
      case 'citizen': return 'bg-green-100 text-green-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800'
      case 'inactive': return 'bg-red-100 text-red-800'
      case 'suspended': return 'bg-yellow-100 text-yellow-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const handleLogout = () => {
    localStorage.removeItem('admin_session')
    document.cookie = 'auth-token=; path=/; expires=Thu, 01 Jan 1970 00:00:01 GMT'
    router.push('/admin/login')
  }

  if (!adminData) {
    return null
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-white border-b border-slate-200">
        <div className="px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <div className="h-8 w-8 rounded-full bg-primary flex items-center justify-center mr-3">
                <UsersIcon className="h-4 w-4 text-primary-foreground" />
              </div>
              <div>
                <h1 className="text-xl font-semibold text-slate-900">Manajemen Pengguna</h1>
                <p className="text-xs text-slate-500">Kelola akun pengguna website</p>
              </div>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-right">
                <p className="text-sm font-medium text-slate-900">{adminData?.name}</p>
                <p className="text-xs text-slate-500">Online</p>
              </div>
              <Button>
                <User className="h-4 w-4 mr-2" />
                Tambah Pengguna
              </Button>
              <Button variant="outline" size="sm" onClick={handleLogout}>
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar */}
        <aside className="w-64 bg-white border-r border-slate-200 min-h-screen">
          <nav className="p-4 space-y-2">
            <Link href="/admin/dashboard" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <BarChart3 className="h-5 w-5" />
              <span>Dashboard</span>
            </Link>
            <Link href="/admin/articles" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <FileText className="h-5 w-5" />
              <span>Artikel</span>
            </Link>
            <Link href="/admin/reports" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <MessageSquare className="h-5 w-5" />
              <span>Laporan</span>
            </Link>
            <Link href="/admin/users" className="flex items-center space-x-3 px-3 py-2 rounded-lg bg-primary text-primary-foreground">
              <UsersIcon className="h-5 w-5" />
              <span>Pengguna</span>
            </Link>
            <Link href="/admin/calendar" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <CalendarIcon className="h-5 w-5" />
              <span>Agenda</span>
            </Link>
            <Link href="/admin/settings" className="flex items-center space-x-3 px-3 py-2 rounded-lg hover:bg-slate-100 text-slate-700">
              <Settings className="h-5 w-5" />
              <span>Pengaturan</span>
            </Link>
          </nav>
        </aside>

        {/* Main Content */}
        <main className="flex-1 p-6">
          {/* Filters */}
          <Card className="mb-6">
            <CardContent className="p-4">
              <div className="flex flex-col lg:flex-row gap-4">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-slate-400" />
                  <Input
                    placeholder="Cari pengguna..."
                    className="pl-10"
                  />
                </div>
                
                <select className="px-3 py-2 border rounded-lg bg-white">
                  <option value="all">Semua Role</option>
                  <option value="citizen">Warga</option>
                  <option value="admin">Admin</option>
                  <option value="moderator">Moderator</option>
                </select>

                <select className="px-3 py-2 border rounded-lg bg-white">
                  <option value="all">Semua Status</option>
                  <option value="active">Aktif</option>
                  <option value="inactive">Non-aktif</option>
                  <option value="suspended">Diblokir</option>
                </select>

                <Button variant="outline">
                  <Filter className="h-4 w-4 mr-2" />
                  Filter
                </Button>
              </div>
            </CardContent>
          </Card>

          {/* Users List */}
          <div className="space-y-4">
            {users.map((user: any) => (
              <Card key={user.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-6">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center space-x-3 mb-2">
                        <div className="h-10 w-10 rounded-full bg-primary flex items-center justify-center">
                          <User className="h-5 w-5 text-primary-foreground" />
                        </div>
                        <div>
                          <h3 className="text-lg font-semibold">{user.name}</h3>
                          <div className="flex items-center space-x-2">
                            <Badge className={getRoleColor(user.role)}>
                              {user.role}
                            </Badge>
                            <Badge className={getStatusColor(user.status)}>
                              {user.status}
                            </Badge>
                          </div>
                        </div>
                      </div>
                      
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
                        <div className="flex items-center text-sm text-slate-600">
                          <Mail className="h-4 w-4 mr-2" />
                          {user.email}
                        </div>
                        <div className="flex items-center text-sm text-slate-600">
                          <Calendar className="h-4 w-4 mr-2" />
                          Bergabung: {user.joinedDate}
                        </div>
                        <div className="flex items-center text-sm text-slate-600">
                          <Shield className="h-4 w-4 mr-2" />
                          Laporan: {user.reportsCount}
                        </div>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-2 ml-4">
                      <Button variant="outline" size="sm">
                        <User className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {users.length === 0 && (
            <Card>
              <CardContent className="p-12 text-center">
                <p className="text-slate-500">Tidak ada pengguna yang ditemukan.</p>
                <Button className="mt-4">
                  <User className="h-4 w-4 mr-2" />
                  Tambah Pengguna Baru
                </Button>
              </CardContent>
            </Card>
          )}
        </main>
      </div>
    </div>
  )
}