'use client'

import { useState } from 'react'
import Header from '@/components/layout/Header'
import Footer from '@/components/layout/Footer'
import { Card, CardContent } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Calendar, MapPin, Clock, Filter, Search, ArrowRight, Users } from 'lucide-react'

export default function AgendaPage() {
  const [searchTerm, setSearchTerm] = useState('')
  const [selectedType, setSelectedType] = useState('all')
  const [selectedMonth, setSelectedMonth] = useState('all')

  // Sample data - akan diganti dengan data dari database
  const agenda = [
    {
      id: 1,
      title: "Reses Masyarakat di Jakarta Selatan",
      description: "Menjaring aspirasi masyarakat dan menyerap keluhan langsung dari warga Jakarta Selatan. Akan hadir para tokoh masyarakat dan perwakilan warga.",
      date: "2024-01-15",
      time: "09:00 - 12:00",
      location: "Kantor Kecamatan Kebayoran Lama",
      type: "Reses",
      status: "upcoming",
      attendees: 150,
      maxAttendees: 200
    },
    {
      id: 2,
      title: "Rapat Kerja Komisi X DPR RI",
      description: "Pembahasan anggaran pendidikan tahun 2024 bersama Kementerian Pendidikan dan Kebudayaan",
      date: "2024-01-18",
      time: "13:00 - 16:00",
      location: "Gedung DPR RI, Jakarta",
      type: "Rapat",
      status: "upcoming",
      attendees: 50,
      maxAttendees: 100
    },
    {
      id: 3,
      title: "Sosialisasi Program Kesehatan Gratis",
      description: "Mengenalkan program kesehatan gratis kepada masyarakat dan menjawab berbagai pertanyaan terkait implementasinya",
      date: "2024-01-20",
      time: "10:00 - 14:00",
      location: "Puskesmas Kecamatan Mampang",
      type: "Sosialisasi",
      status: "upcoming",
      attendees: 75,
      maxAttendees: 150
    },
    {
      id: 4,
      title: "Kunjungan Kerja ke Pabrik Tekstil",
      description: "Meninjau kondisi industri tekstil dan berdialog dengan pengusaha serta pekerja terkait isu ketenagakerjaan",
      date: "2024-01-08",
      time: "09:00 - 15:00",
      location: "Kawasan Industri Tangerang",
      type: "Kunjungan",
      status: "completed",
      attendees: 30,
      maxAttendees: 50
    },
    {
      id: 5,
      title: "Kampanye Damai Pemilu 2024",
      description: "Mengajak masyarakat untuk menggunakan hak pilih secara damai dan bertanggung jawab",
      date: "2024-01-25",
      time: "16:00 - 19:00",
      location: "Alun-Alun Kota Tangerang",
      type: "Kampaye",
      status: "upcoming",
      attendees: 500,
      maxAttendees: 1000
    }
  ]

  const typeColors: Record<string, string> = {
    "Reses": "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
    "Rapat": "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
    "Sosialisasi": "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
    "Kunjungan": "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
    "Kampaye": "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300"
  }

  const statusColors: Record<string, string> = {
    "upcoming": "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300",
    "ongoing": "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
    "completed": "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
    "cancelled": "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300"
  }

  const statusLabels: Record<string, string> = {
    "upcoming": "Akan Datang",
    "ongoing": "Sedang Berlangsung",
    "completed": "Selesai",
    "cancelled": "Dibatalkan"
  }

  const formatDate = (dateString: string) => {
    const date = new Date(dateString)
    const options: Intl.DateTimeFormatOptions = { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    }
    return date.toLocaleDateString('id-ID', options)
  }

  const isToday = (dateString: string) => {
    const today = new Date()
    const date = new Date(dateString)
    return date.toDateString() === today.toDateString()
  }

  const isPast = (dateString: string) => {
    const today = new Date()
    const date = new Date(dateString)
    return date < today
  }

  const filteredAgenda = agenda.filter(item => {
    const matchesSearch = item.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         item.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         item.location.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesType = selectedType === 'all' || item.type === selectedType
    const matchesMonth = selectedMonth === 'all' || new Date(item.date).toLocaleDateString('id-ID', { month: 'long' }) === selectedMonth
    
    return matchesSearch && matchesType && matchesMonth
  })

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-primary/10 to-secondary/10 py-16">
        <div className="container mx-auto px-4">
          <div className="text-center">
            <h1 className="text-4xl md:text-5xl font-bold mb-4">
              Agenda & Kegiatan
            </h1>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Jadwal lengkap kegiatan dan program kerja. Ikuti dan berpartisipasi dalam kegiatan kami.
            </p>
          </div>
        </div>
      </section>

      {/* Filters */}
      <section className="py-8 border-b">
        <div className="container mx-auto px-4">
          <div className="flex flex-col lg:flex-row gap-4 items-center justify-between">
            <div className="flex flex-col sm:flex-row gap-4 flex-1">
              <div className="relative flex-1 max-w-md">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Cari agenda..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
              
              <Select value={selectedType} onValueChange={setSelectedType}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="Tipe Kegiatan" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Semua Tipe</SelectItem>
                  <SelectItem value="Reses">Reses</SelectItem>
                  <SelectItem value="Rapat">Rapat</SelectItem>
                  <SelectItem value="Sosialisasi">Sosialisasi</SelectItem>
                  <SelectItem value="Kunjungan">Kunjungan</SelectItem>
                  <SelectItem value="Kampaye">Kampanye</SelectItem>
                </SelectContent>
              </Select>

              <Select value={selectedMonth} onValueChange={setSelectedMonth}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="Bulan" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Semua Bulan</SelectItem>
                  <SelectItem value="Januari">Januari</SelectItem>
                  <SelectItem value="Februari">Februari</SelectItem>
                  <SelectItem value="Maret">Maret</SelectItem>
                  <SelectItem value="April">April</SelectItem>
                  <SelectItem value="Mei">Mei</SelectItem>
                  <SelectItem value="Juni">Juni</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <Button variant="outline" className="w-full sm:w-auto">
              <Calendar className="mr-2 h-4 w-4" />
              Kalender
            </Button>
          </div>
        </div>
      </section>

      {/* Agenda List */}
      <section className="py-12">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-6">
            {filteredAgenda.map((item) => {
              const today = isToday(item.date)
              const past = isPast(item.date)
              
              return (
                <Card key={item.id} className={`hover:shadow-lg transition-all duration-300 ${today ? 'ring-2 ring-primary' : ''} ${past ? 'opacity-75' : ''}`}>
                  <CardContent className="p-6">
                    <div className="space-y-4">
                      {/* Header */}
                      <div className="flex items-start justify-between">
                        <div className="space-y-2">
                          <div className="flex items-center space-x-2">
                            <Badge 
                              variant="secondary" 
                              className={typeColors[item.type] || 'bg-gray-100 text-gray-800'}
                            >
                              {item.type}
                            </Badge>
                            {today && (
                              <Badge className="bg-primary text-primary-foreground">
                                Hari Ini
                              </Badge>
                            )}
                            <Badge 
                              variant="outline" 
                              className={statusColors[item.status] || 'bg-gray-100 text-gray-800'}
                            >
                              {statusLabels[item.status]}
                            </Badge>
                          </div>
                          <h3 className="text-xl font-semibold line-clamp-2">
                            {item.title}
                          </h3>
                        </div>
                      </div>

                      {/* Description */}
                      <p className="text-muted-foreground line-clamp-3">
                        {item.description}
                      </p>

                      {/* Details */}
                      <div className="space-y-2">
                        <div className="flex items-center text-sm text-muted-foreground">
                          <Calendar className="h-4 w-4 mr-2 flex-shrink-0" />
                          <span>{formatDate(item.date)}</span>
                        </div>
                        <div className="flex items-center text-sm text-muted-foreground">
                          <Clock className="h-4 w-4 mr-2 flex-shrink-0" />
                          <span>{item.time}</span>
                        </div>
                        <div className="flex items-center text-sm text-muted-foreground">
                          <MapPin className="h-4 w-4 mr-2 flex-shrink-0" />
                          <span className="line-clamp-1">{item.location}</span>
                        </div>
                        <div className="flex items-center text-sm text-muted-foreground">
                          <Users className="h-4 w-4 mr-2 flex-shrink-0" />
                          <span>{item.attendees} / {item.maxAttendees} Peserta</span>
                        </div>
                      </div>

                      {/* Progress Bar */}
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div 
                          className="bg-primary h-2 rounded-full transition-all duration-300"
                          style={{ width: `${(item.attendees / item.maxAttendees) * 100}%` }}
                        />
                      </div>

                      {/* Actions */}
                      <div className="pt-2">
                        {!past && (
                          <div className="flex space-x-2">
                            <Button size="sm" className="flex-1">
                              {today ? 'Gabung Sekarang' : 'Daftar'}
                            </Button>
                            <Button size="sm" variant="outline">
                              <ArrowRight className="h-4 w-4" />
                            </Button>
                          </div>
                        )}
                        {past && (
                          <Button size="sm" variant="outline" className="w-full">
                            Lihat Dokumentasi
                          </Button>
                        )}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              )
            })}
          </div>

          {filteredAgenda.length === 0 && (
            <div className="text-center py-12">
              <Calendar className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
              <h3 className="text-lg font-semibold mb-2">Tidak Ada Agenda</h3>
              <p className="text-muted-foreground">
                Tidak ada agenda yang ditemukan dengan filter yang dipilih.
              </p>
            </div>
          )}

          {/* Load More */}
          {filteredAgenda.length > 0 && (
            <div className="text-center mt-12">
              <Button variant="outline">
                Muat Lebih Banyak
              </Button>
            </div>
          )}
        </div>
      </section>

      {/* Calendar Integration */}
      <section className="py-16 bg-muted/30">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto text-center">
            <h2 className="text-3xl font-bold mb-4">
              Sinkronisasi Kalender
            </h2>
            <p className="text-lg text-muted-foreground mb-8">
              Jangan ketinggalan kegiatan penting. Sinkronkan agenda kami dengan kalender Anda.
            </p>
            
            <div className="grid md:grid-cols-3 gap-4">
              <Card className="p-6 hover:shadow-lg transition-shadow cursor-pointer">
                <div className="text-4xl mb-4">📅</div>
                <h3 className="font-semibold mb-2">Google Calendar</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Tambahkan ke Google Calendar
                </p>
                <Button size="sm" variant="outline" className="w-full">
                  Hubungkan
                </Button>
              </Card>
              
              <Card className="p-6 hover:shadow-lg transition-shadow cursor-pointer">
                <div className="text-4xl mb-4">📱</div>
                <h3 className="font-semibold mb-2">Apple Calendar</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Download file .ics
                </p>
                <Button size="sm" variant="outline" className="w-full">
                  Download
                </Button>
              </Card>
              
              <Card className="p-6 hover:shadow-lg transition-shadow cursor-pointer">
                <div className="text-4xl mb-4">📧</div>
                <h3 className="font-semibold mb-2">Email Reminder</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Dapatkan pengingat via email
                </p>
                <Button size="sm" variant="outline" className="w-full">
                  Berlangganan
                </Button>
              </Card>
            </div>
          </div>
        </div>
      </section>
      
      <Footer />
    </div>
  )
}