import { NextRequest, NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const page = parseInt(searchParams.get('page') || '1')
    const limit = parseInt(searchParams.get('limit') || '10')
    const type = searchParams.get('type')
    const search = searchParams.get('search')
    const upcoming = searchParams.get('upcoming')

    // Build where clause
    let whereClause: any = {}
    
    if (type && type !== 'all') {
      whereClause.type = type
    }
    
    if (search) {
      whereClause.OR = [
        { title: { contains: search } },
        { description: { contains: search } },
        { location: { contains: search } }
      ]
    }
    
    if (upcoming === 'true') {
      whereClause.startDate = {
        gte: new Date()
      }
    }

    // Get total count
    const total = await db.agenda.count({ where: whereClause })

    // Get agenda with pagination
    const agendas = await db.agenda.findMany({
      where: whereClause,
      orderBy: { startDate: 'asc' },
      skip: (page - 1) * limit,
      take: limit,
      include: {
        organizer: {
          select: {
            id: true,
            name: true,
            email: true,
            avatar: true
          }
        }
      }
    })

    return NextResponse.json({
      success: true,
      data: agendas,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit)
      }
    })
  } catch (error) {
    console.error('Error fetching agendas:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to fetch agendas' },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    
    const {
      title,
      description,
      location,
      startDate,
      endDate,
      type,
      organizerId,
      attachment,
      published = false
    } = body

    // Validate required fields
    if (!title || !startDate || !organizerId) {
      return NextResponse.json(
        { success: false, error: 'Missing required fields' },
        { status: 400 }
      )
    }

    // Create new agenda
    const agenda = await db.agenda.create({
      data: {
        title,
        description,
        location,
        startDate: new Date(startDate),
        endDate: endDate ? new Date(endDate) : null,
        type,
        organizerId,
        attachment,
        published
      }
    })

    return NextResponse.json({
      success: true,
      data: agenda,
      message: 'Agenda created successfully'
    })
  } catch (error) {
    console.error('Error creating agenda:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to create agenda' },
      { status: 500 }
    )
  }
}