'use client'

import { useState } from 'react'
import Header from '@/components/layout/Header'
import Footer from '@/components/layout/Footer'
import { Card, CardContent } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Calendar, Clock, User, Search, Filter, ArrowRight, Eye, Share2, Heart } from 'lucide-react'

export default function BeritaPage() {
  const [searchTerm, setSearchTerm] = useState('')
  const [selectedCategory, setSelectedCategory] = useState('all')
  const [sortBy, setSortBy] = useState('latest')

  // Sample data - akan diganti dengan data dari database
  const articles = [
    {
      id: 1,
      title: "Program Bantuan UMKM Sukses Diluncurkan di Jakarta",
      excerpt: "Program bantuan modal usaha untuk pelaku UMKM telah resmi diluncurkan dan mendapat sambutan positif dari masyarakat. Lebih dari 500 pelaku usaha telah mendaftar dalam minggu pertama.",
      content: "Program bantuan modal usaha untuk pelaku UMKM telah resmi diluncurkan dan mendapat sambutan positif dari masyarakat. Lebih dari 500 pelaku usaha telah mendaftar dalam minggu pertama.",
      category: "Ekonomi",
      author: "Tim Media",
      date: "2024-01-15",
      readTime: "3 menit",
      thumbnail: "/api/placeholder/400/250",
      featured: true,
      viewCount: 1250,
      likeCount: 89
    },
    {
      id: 2,
      title: "Kunjungan ke Pusat Kesehatan Masyarakat di Bekasi",
      excerpt: "Meninjau langsung kondisi fasilitas kesehatan dan berdialog dengan tenaga medis serta pasien tentang pelayanan kesehatan di wilayah Bekasi.",
      content: "Meninjau langsung kondisi fasilitas kesehatan dan berdialog dengan tenaga medis serta pasien tentang pelayanan kesehatan di wilayah Bekasi.",
      category: "Kesehatan",
      author: "Tim Media",
      date: "2024-01-14",
      readTime: "4 menit",
      thumbnail: "/api/placeholder/400/250",
      featured: false,
      viewCount: 890,
      likeCount: 67
    },
    {
      id: 3,
      title: "Sosialisasi Program Pendidikan Gratis di Tangerang",
      excerpt: "Mengenalkan program pendidikan gratis kepada masyarakat dan menjawab berbagai pertanyaan terkait implementasi program di tahun ajaran baru.",
      content: "Mengenalkan program pendidikan gratis kepada masyarakat dan menjawab berbagai pertanyaan terkait implementasi program di tahun ajaran baru.",
      category: "Pendidikan",
      author: "Tim Media",
      date: "2024-01-13",
      readTime: "5 menit",
      thumbnail: "/api/placeholder/400/250",
      featured: false,
      viewCount: 1100,
      likeCount: 92
    },
    {
      id: 4,
      title: "Peresmian Jalan Baru di Bogor",
      excerpt: "Meresmikan jalan baru yang menghubungkan dua kecamatan di Bogor untuk memperlancar akses transportasi dan perekonomian warga.",
      content: "Meresmikan jalan baru yang menghubungkan dua kecamatan di Bogor untuk memperlancar akses transportasi dan perekonomian warga.",
      category: "Infrastruktur",
      author: "Tim Media",
      date: "2024-01-12",
      readTime: "3 menit",
      thumbnail: "/api/placeholder/400/250",
      featured: false,
      viewCount: 750,
      likeCount: 45
    },
    {
      id: 5,
      title: "Bantuan Sosial untuk Korban Banjir",
      excerpt: "Menyalurkan bantuan sosial kepada korban banjir di wilayah Jakarta dan sekitarnya, bekerja sama dengan berbagai pihak terkait.",
      content: "Menyalurkan bantuan sosial kepada korban banjir di wilayah Jakarta dan sekitarnya, bekerja sama dengan berbagai pihak terkait.",
      category: "Sosial",
      author: "Tim Media",
      date: "2024-01-11",
      readTime: "4 menit",
      thumbnail: "/api/placeholder/400/250",
      featured: false,
      viewCount: 1500,
      likeCount: 120
    },
    {
      id: 6,
      title: "Rapat Koordinasi dengan Kepala Desa",
      excerpt: "Melakukan rapat koordinasi dengan para kepala desa untuk membahas program pembangunan desa dan isu-isu strategis lainnya.",
      content: "Melakukan rapat koordinasi dengan para kepala desa untuk membahas program pembangunan desa dan isu-isu strategis lainnya.",
      category: "Pemerintahan",
      author: "Tim Media",
      date: "2024-01-10",
      readTime: "6 menit",
      thumbnail: "/api/placeholder/400/250",
      featured: false,
      viewCount: 620,
      likeCount: 38
    }
  ]

  const categories = [
    { value: "all", label: "Semua Kategori" },
    { value: "Ekonomi", label: "Ekonomi" },
    { value: "Kesehatan", label: "Kesehatan" },
    { value: "Pendidikan", label: "Pendidikan" },
    { value: "Infrastruktur", label: "Infrastruktur" },
    { value: "Sosial", label: "Sosial" },
    { value: "Pemerintahan", label: "Pemerintahan" }
  ]

  const categoryColors: Record<string, string> = {
    "Ekonomi": "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
    "Kesehatan": "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300",
    "Pendidikan": "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
    "Sosial": "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
    "Infrastruktur": "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
    "Pemerintahan": "bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300"
  }

  const formatDate = (dateString: string) => {
    const date = new Date(dateString)
    const options: Intl.DateTimeFormatOptions = { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    }
    return date.toLocaleDateString('id-ID', options)
  }

  const formatViewCount = (count: number) => {
    if (count >= 1000) {
      return `${(count / 1000).toFixed(1)}K`
    }
    return count.toString()
  }

  const filteredArticles = articles.filter(article => {
    const matchesSearch = article.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         article.excerpt.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesCategory = selectedCategory === 'all' || article.category === selectedCategory
    
    return matchesSearch && matchesCategory
  }).sort((a, b) => {
    if (sortBy === 'latest') {
      return new Date(b.date).getTime() - new Date(a.date).getTime()
    } else if (sortBy === 'popular') {
      return b.viewCount - a.viewCount
    }
    return 0
  })

  const featuredArticle = filteredArticles.find(article => article.featured)
  const regularArticles = filteredArticles.filter(article => !article.featured)

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-primary/10 to-secondary/10 py-16">
        <div className="container mx-auto px-4">
          <div className="text-center">
            <h1 className="text-4xl md:text-5xl font-bold mb-4">
              Berita & Artikel
            </h1>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Update terkini kegiatan, program kerja, dan informasi penting lainnya. 
              Tetap terhubung dengan kami melalui berita terbaru.
            </p>
          </div>
        </div>
      </section>

      {/* Filters */}
      <section className="py-8 border-b">
        <div className="container mx-auto px-4">
          <div className="flex flex-col lg:flex-row gap-4 items-center justify-between">
            <div className="flex flex-col sm:flex-row gap-4 flex-1">
              <div className="relative flex-1 max-w-md">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Cari berita..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
              
              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="Kategori" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.value} value={category.value}>
                      {category.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>

              <Select value={sortBy} onValueChange={setSortBy}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="Urutkan" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="latest">Terbaru</SelectItem>
                  <SelectItem value="popular">Terpopuler</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <Button variant="outline" className="w-full sm:w-auto">
              <Filter className="mr-2 h-4 w-4" />
              Filter Lainnya
            </Button>
          </div>
        </div>
      </section>

      {/* Featured Article */}
      {featuredArticle && (
        <section className="py-12">
          <div className="container mx-auto px-4">
            <h2 className="text-2xl font-bold mb-8">Artikel Utama</h2>
            <Card className="overflow-hidden hover:shadow-lg transition-shadow">
              <div className="grid lg:grid-cols-2 gap-8">
                <div className="relative aspect-video lg:aspect-square">
                  <div className="w-full h-full bg-gradient-to-br from-primary/20 to-secondary/20 flex items-center justify-center">
                    <div className="text-center text-muted-foreground">
                      <div className="w-20 h-20 mx-auto mb-4 rounded-full bg-muted flex items-center justify-center">
                        📰
                      </div>
                      <p className="text-lg">Gambar Artikel Utama</p>
                    </div>
                  </div>
                  <Badge className="absolute top-4 left-4 bg-primary text-primary-foreground">
                    Utama
                  </Badge>
                </div>
                
                <CardContent className="p-8 flex flex-col justify-between">
                  <div className="space-y-4">
                    <div className="flex items-center space-x-2">
                      <Badge 
                        variant="secondary" 
                        className={categoryColors[featuredArticle.category]}
                      >
                        {featuredArticle.category}
                      </Badge>
                    </div>
                    
                    <h3 className="text-3xl font-bold line-clamp-2">
                      {featuredArticle.title}
                    </h3>
                    
                    <p className="text-muted-foreground line-clamp-3">
                      {featuredArticle.excerpt}
                    </p>
                    
                    <div className="flex items-center justify-between text-sm text-muted-foreground">
                      <div className="flex items-center space-x-4">
                        <div className="flex items-center space-x-1">
                          <User className="h-4 w-4" />
                          <span>{featuredArticle.author}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Clock className="h-4 w-4" />
                          <span>{featuredArticle.readTime}</span>
                        </div>
                      </div>
                      <div className="flex items-center space-x-4">
                        <div className="flex items-center space-x-1">
                          <Eye className="h-4 w-4" />
                          <span>{formatViewCount(featuredArticle.viewCount)}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Heart className="h-4 w-4" />
                          <span>{featuredArticle.likeCount}</span>
                        </div>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex items-center justify-between pt-4">
                    <div className="text-sm text-muted-foreground">
                      {formatDate(featuredArticle.date)}
                    </div>
                    <Button>
                      Baca Selengkapnya
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </Button>
                  </div>
                </CardContent>
              </div>
            </Card>
          </div>
        </section>
      )}

      {/* Articles Grid */}
      <section className="py-12">
        <div className="container mx-auto px-4">
          <h2 className="text-2xl font-bold mb-8">
            {featuredArticle ? 'Berita Lainnya' : 'Semua Berita'}
          </h2>
          
          <div className="grid lg:grid-cols-3 md:grid-cols-2 gap-8">
            {regularArticles.map((article) => (
              <Card key={article.id} className="group hover:shadow-lg transition-all duration-300 overflow-hidden">
                {/* Thumbnail */}
                <div className="relative aspect-video overflow-hidden">
                  <div className="w-full h-full bg-gradient-to-br from-primary/20 to-secondary/20 flex items-center justify-center">
                    <div className="text-center text-muted-foreground">
                      <div className="w-12 h-12 mx-auto mb-2 rounded-full bg-muted flex items-center justify-center">
                        📄
                      </div>
                      <p className="text-sm">Gambar Berita</p>
                    </div>
                  </div>

                  {/* Category Badge */}
                  <Badge 
                    variant="secondary" 
                    className={`absolute top-4 right-4 ${categoryColors[article.category]}`}
                  >
                    {article.category}
                  </Badge>
                </div>

                <CardContent className="p-6">
                  <div className="space-y-4">
                    {/* Meta Info */}
                    <div className="flex items-center justify-between text-sm text-muted-foreground">
                      <div className="flex items-center space-x-3">
                        <div className="flex items-center space-x-1">
                          <User className="h-3 w-3" />
                          <span>{article.author}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Clock className="h-3 w-3" />
                          <span>{article.readTime}</span>
                        </div>
                      </div>
                      <div className="flex items-center space-x-2">
                        <div className="flex items-center space-x-1">
                          <Eye className="h-3 w-3" />
                          <span>{formatViewCount(article.viewCount)}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Heart className="h-3 w-3" />
                          <span>{article.likeCount}</span>
                        </div>
                      </div>
                    </div>

                    {/* Content */}
                    <div className="space-y-2">
                      <h3 className="text-xl font-semibold line-clamp-2 group-hover:text-primary transition-colors">
                        {article.title}
                      </h3>
                      <p className="text-muted-foreground line-clamp-3">
                        {article.excerpt}
                      </p>
                    </div>

                    {/* Footer */}
                    <div className="flex items-center justify-between pt-2">
                      <div className="text-sm text-muted-foreground">
                        {formatDate(article.date)}
                      </div>
                      <div className="flex space-x-2">
                        <Button size="sm" variant="ghost">
                          <Share2 className="h-4 w-4" />
                        </Button>
                        <Button size="sm">
                          <ArrowRight className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {filteredArticles.length === 0 && (
            <div className="text-center py-12">
              <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-muted flex items-center justify-center">
                <Search className="h-8 w-8 text-muted-foreground" />
              </div>
              <h3 className="text-lg font-semibold mb-2">Tidak Ada Berita</h3>
              <p className="text-muted-foreground">
                Tidak ada berita yang ditemukan dengan filter yang dipilih.
              </p>
            </div>
          )}

          {/* Load More */}
          {filteredArticles.length > 0 && (
            <div className="text-center mt-12">
              <Button variant="outline">
                Muat Lebih Banyak
              </Button>
            </div>
          )}
        </div>
      </section>

      {/* Newsletter */}
      <section className="py-16 bg-muted/30">
        <div className="container mx-auto px-4">
          <div className="max-w-2xl mx-auto text-center">
            <h2 className="text-3xl font-bold mb-4">
              Berlangganan Newsletter
            </h2>
            <p className="text-lg text-muted-foreground mb-8">
              Dapatkan berita terbaru dan update kegiatan langsung di inbox Anda
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 max-w-md mx-auto">
              <Input
                type="email"
                placeholder="Masukkan email Anda"
                className="flex-1"
              />
              <Button>
                Berlangganan
              </Button>
            </div>
          </div>
        </div>
      </section>
      
      <Footer />
    </div>
  )
}