'use client'

import { useState } from 'react'
import Header from '@/components/layout/Header'
import Footer from '@/components/layout/Footer'
import { Card, CardContent } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Dialog, DialogContent, DialogTrigger } from '@/components/ui/dialog'
import { Search, Filter, Play, Image as ImageIcon, Calendar, Eye, Heart, Download, Share2, X } from 'lucide-react'

export default function GaleriPage() {
  const [searchTerm, setSearchTerm] = useState('')
  const [selectedType, setSelectedType] = useState('all')
  const [selectedCategory, setSelectedCategory] = useState('all')
  const [selectedMedia, setSelectedMedia] = useState<any>(null)

  // Sample data - akan diganti dengan data dari database
  const gallery = [
    {
      id: 1,
      title: "Kunjungan ke Puskesmas",
      description: "Kunjungan kerja ke Puskesmas Kecamatan Mampang untuk meninjau pelayanan kesehatan masyarakat",
      type: "image",
      category: "Kunjungan",
      url: "/api/placeholder/800/600",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-15",
      viewCount: 1250,
      likeCount: 89
    },
    {
      id: 2,
      title: "Sosialisasi Program Pendidikan",
      description: "Sosialisasi program pendidikan gratis di SDN 01 Jakarta",
      type: "video",
      category: "Sosialisasi",
      url: "https://www.youtube.com/embed/dQw4w9WgXcQ",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-14",
      viewCount: 2100,
      likeCount: 156
    },
    {
      id: 3,
      title: "Reses Masyarakat Jakarta Selatan",
      description: "Reses menyerap aspirasi masyarakat Jakarta Selatan",
      type: "image",
      category: "Reses",
      url: "/api/placeholder/800/600",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-13",
      viewCount: 980,
      likeCount: 67
    },
    {
      id: 4,
      title: "Peresmian Jalan Baru",
      description: "Peresmian jalan baru yang menghubungkan dua kecamatan di Bogor",
      type: "video",
      category: "Infrastruktur",
      url: "https://www.youtube.com/embed/dQw4w9WgXcQ",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-12",
      viewCount: 3200,
      likeCount: 234
    },
    {
      id: 5,
      title: "Bantuan Sosial Korban Banjir",
      description: "Penyaluran bantuan sosial kepada korban banjir di Jakarta",
      type: "image",
      category: "Sosial",
      url: "/api/placeholder/800/600",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-11",
      viewCount: 1500,
      likeCount: 120
    },
    {
      id: 6,
      title: "Rapat Koordinasi Kepala Desa",
      description: "Rapat koordinasi dengan para kepala desa se-Jawa Barat",
      type: "image",
      category: "Rapat",
      url: "/api/placeholder/800/600",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-10",
      viewCount: 750,
      likeCount: 45
    },
    {
      id: 7,
      title: "Kampanye Pemilu Damai",
      description: "Kampanye damai pemilu 2024 di Alun-Alun Tangerang",
      type: "video",
      category: "Kampanye",
      url: "https://www.youtube.com/embed/dQw4w9WgXcQ",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-09",
      viewCount: 4500,
      likeCount: 320
    },
    {
      id: 8,
      title: "Kunjungan ke Pabrik Tekstil",
      description: "Kunjungan ke pabrik tekstil di Kawasan Industri Tangerang",
      type: "image",
      category: "Kunjungan",
      url: "/api/placeholder/800/600",
      thumbnail: "/api/placeholder/400/300",
      date: "2024-01-08",
      viewCount: 890,
      likeCount: 78
    }
  ]

  const categories = [
    { value: "all", label: "Semua Kategori" },
    { value: "Kunjungan", label: "Kunjungan" },
    { value: "Sosialisasi", label: "Sosialisasi" },
    { value: "Reses", label: "Reses" },
    { value: "Infrastruktur", label: "Infrastruktur" },
    { value: "Sosial", label: "Sosial" },
    { value: "Rapat", label: "Rapat" },
    { value: "Kampanye", label: "Kampanye" }
  ]

  const categoryColors: Record<string, string> = {
    "Kunjungan": "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
    "Sosialisasi": "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
    "Reses": "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
    "Infrastruktur": "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
    "Sosial": "bg-pink-100 text-pink-800 dark:bg-pink-900 dark:text-pink-300",
    "Rapat": "bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300",
    "Kampanye": "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300"
  }

  const formatDate = (dateString: string) => {
    const date = new Date(dateString)
    const options: Intl.DateTimeFormatOptions = { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    }
    return date.toLocaleDateString('id-ID', options)
  }

  const formatViewCount = (count: number) => {
    if (count >= 1000) {
      return `${(count / 1000).toFixed(1)}K`
    }
    return count.toString()
  }

  const filteredGallery = gallery.filter(item => {
    const matchesSearch = item.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         item.description.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesType = selectedType === 'all' || item.type === selectedType
    const matchesCategory = selectedCategory === 'all' || item.category === selectedCategory
    
    return matchesSearch && matchesType && matchesCategory
  })

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-primary/10 to-secondary/10 py-16">
        <div className="container mx-auto px-4">
          <div className="text-center">
            <h1 className="text-4xl md:text-5xl font-bold mb-4">
              Galeri Foto & Video
            </h1>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Dokumentasi lengkap kegiatan dan program kerja. Lihat momen-momen penting 
              dalam bentuk foto dan video berkualitas tinggi.
            </p>
          </div>
        </div>
      </section>

      {/* Filters */}
      <section className="py-8 border-b">
        <div className="container mx-auto px-4">
          <div className="flex flex-col lg:flex-row gap-4 items-center justify-between">
            <div className="flex flex-col sm:flex-row gap-4 flex-1">
              <div className="relative flex-1 max-w-md">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Cari di galeri..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
              
              <Select value={selectedType} onValueChange={setSelectedType}>
                <SelectTrigger className="w-full sm:w-40">
                  <SelectValue placeholder="Tipe" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Semua</SelectItem>
                  <SelectItem value="image">Foto</SelectItem>
                  <SelectItem value="video">Video</SelectItem>
                </SelectContent>
              </Select>

              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder="Kategori" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.value} value={category.value}>
                      {category.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <Button variant="outline" className="w-full sm:w-auto">
              <Filter className="mr-2 h-4 w-4" />
              Filter Lainnya
            </Button>
          </div>
        </div>
      </section>

      {/* Gallery Grid */}
      <section className="py-12">
        <div className="container mx-auto px-4">
          <div className="grid sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {filteredGallery.map((item) => (
              <Card key={item.id} className="group overflow-hidden hover:shadow-lg transition-all duration-300">
                <div className="relative aspect-square">
                  {/* Thumbnail */}
                  <div className="w-full h-full bg-gradient-to-br from-primary/20 to-secondary/20 flex items-center justify-center">
                    {item.type === 'image' ? (
                      <ImageIcon className="w-12 h-12 text-muted-foreground" />
                    ) : (
                      <Play className="w-12 h-12 text-muted-foreground" />
                    )}
                  </div>
                  
                  {/* Type Badge */}
                  <div className="absolute top-2 left-2">
                    <Badge variant="secondary" className="bg-black/70 text-white">
                      {item.type === 'image' ? 'Foto' : 'Video'}
                    </Badge>
                  </div>

                  {/* Category Badge */}
                  <div className="absolute top-2 right-2">
                    <Badge 
                      variant="secondary" 
                      className={categoryColors[item.category]}
                    >
                      {item.category}
                    </Badge>
                  </div>

                  {/* Play Button for Videos */}
                  {item.type === 'video' && (
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="w-16 h-16 rounded-full bg-white/90 flex items-center justify-center group-hover:scale-110 transition-transform">
                        <Play className="h-6 w-6 text-primary ml-1" />
                      </div>
                    </div>
                  )}

                  {/* Hover Overlay */}
                  <div className="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                    <div className="flex space-x-2">
                      <Button size="sm" variant="secondary" onClick={() => setSelectedMedia(item)}>
                        <Eye className="h-4 w-4" />
                      </Button>
                      <Button size="sm" variant="secondary">
                        <Download className="h-4 w-4" />
                      </Button>
                      <Button size="sm" variant="secondary">
                        <Share2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </div>

                <CardContent className="p-4">
                  <div className="space-y-2">
                    <h3 className="font-semibold line-clamp-2 group-hover:text-primary transition-colors">
                      {item.title}
                    </h3>
                    <p className="text-sm text-muted-foreground line-clamp-2">
                      {item.description}
                    </p>
                    
                    <div className="flex items-center justify-between text-xs text-muted-foreground pt-2">
                      <div className="flex items-center space-x-3">
                        <div className="flex items-center space-x-1">
                          <Eye className="h-3 w-3" />
                          <span>{formatViewCount(item.viewCount)}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Heart className="h-3 w-3" />
                          <span>{item.likeCount}</span>
                        </div>
                      </div>
                      <div className="flex items-center space-x-1">
                        <Calendar className="h-3 w-3" />
                        <span>{new Date(item.date).toLocaleDateString('id-ID', { day: 'numeric', month: 'short' })}</span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {filteredGallery.length === 0 && (
            <div className="text-center py-12">
              <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-muted flex items-center justify-center">
                <ImageIcon className="h-8 w-8 text-muted-foreground" />
              </div>
              <h3 className="text-lg font-semibold mb-2">Tidak Ada Media</h3>
              <p className="text-muted-foreground">
                Tidak ada foto atau video yang ditemukan dengan filter yang dipilih.
              </p>
            </div>
          )}

          {/* Load More */}
          {filteredGallery.length > 0 && (
            <div className="text-center mt-12">
              <Button variant="outline">
                Muat Lebih Banyak
              </Button>
            </div>
          )}
        </div>
      </section>

      {/* Media Modal */}
      {selectedMedia && (
        <Dialog open={!!selectedMedia} onOpenChange={() => setSelectedMedia(null)}>
          <DialogContent className="max-w-4xl w-full p-0">
            <div className="relative">
              {/* Close Button */}
              <Button
                variant="ghost"
                size="sm"
                className="absolute top-4 right-4 z-10 bg-black/50 text-white hover:bg-black/70"
                onClick={() => setSelectedMedia(null)}
              >
                <X className="h-4 w-4" />
              </Button>

              {/* Media Content */}
              <div className="aspect-video bg-black">
                {selectedMedia.type === 'image' ? (
                  <div className="w-full h-full flex items-center justify-center">
                    <ImageIcon className="w-24 h-24 text-white/50" />
                  </div>
                ) : (
                  <iframe
                    src={selectedMedia.url}
                    className="w-full h-full"
                    allowFullScreen
                  />
                )}
              </div>

              {/* Media Info */}
              <div className="p-6 bg-background">
                <div className="flex items-start justify-between">
                  <div className="space-y-2 flex-1">
                    <div className="flex items-center space-x-2">
                      <Badge 
                        variant="secondary" 
                        className={categoryColors[selectedMedia.category]}
                      >
                        {selectedMedia.category}
                      </Badge>
                      <Badge variant="secondary">
                        {selectedMedia.type === 'image' ? 'Foto' : 'Video'}
                      </Badge>
                    </div>
                    <h2 className="text-2xl font-bold">{selectedMedia.title}</h2>
                    <p className="text-muted-foreground">{selectedMedia.description}</p>
                    <div className="flex items-center space-x-4 text-sm text-muted-foreground">
                      <div className="flex items-center space-x-1">
                        <Calendar className="h-4 w-4" />
                        <span>{formatDate(selectedMedia.date)}</span>
                      </div>
                      <div className="flex items-center space-x-1">
                        <Eye className="h-4 w-4" />
                        <span>{formatViewCount(selectedMedia.viewCount)} views</span>
                      </div>
                      <div className="flex items-center space-x-1">
                        <Heart className="h-4 w-4" />
                        <span>{selectedMedia.likeCount} likes</span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex space-x-2 ml-4">
                    <Button variant="outline" size="sm">
                      <Heart className="h-4 w-4 mr-2" />
                      Suka
                    </Button>
                    <Button variant="outline" size="sm">
                      <Share2 className="h-4 w-4 mr-2" />
                      Bagikan
                    </Button>
                    <Button variant="outline" size="sm">
                      <Download className="h-4 w-4 mr-2" />
                      Download
                    </Button>
                  </div>
                </div>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      )}

      {/* Upload Section */}
      <section className="py-16 bg-muted/30">
        <div className="container mx-auto px-4">
          <div className="max-w-2xl mx-auto text-center">
            <h2 className="text-3xl font-bold mb-4">
              Punya Foto atau Video?
            </h2>
            <p className="text-lg text-muted-foreground mb-8">
              Bagikan momen kegiatan bersama kami. Upload foto atau video dokumentasi kegiatan.
            </p>
            
            <Card className="p-8">
              <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-8 text-center">
                <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-muted flex items-center justify-center">
                  <ImageIcon className="h-8 w-8 text-muted-foreground" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Upload Media</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Seret dan lepas file di sini atau klik untuk browse
                </p>
                <Button>
                  Pilih File
                </Button>
              </div>
            </Card>
          </div>
        </div>
      </section>
      
      <Footer />
    </div>
  )
}