'use client'

import { useState } from 'react'
import Header from '@/components/layout/Header'
import Footer from '@/components/layout/Footer'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Badge } from '@/components/ui/badge'
import { Progress } from '@/components/ui/progress'
import { 
  MessageSquare, 
  Upload, 
  Send, 
  Search, 
  Filter,
  FileText,
  Clock,
  CheckCircle,
  AlertCircle,
  XCircle,
  Calendar,
  MapPin,
  User,
  Mail,
  Phone
} from 'lucide-react'

export default function LaporanPage() {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    region: 'all',
    category: 'all',
    title: '',
    content: '',
    attachment: null
  })
  
  const [searchTerm, setSearchTerm] = useState('')
  const [selectedStatus, setSelectedStatus] = useState('all')
  const [selectedCategory, setSelectedCategory] = useState('all')
  const [isSubmitting, setIsSubmitting] = useState(false)

  // Sample data - akan diganti dengan data dari database
  const reports = [
    {
      id: 'LAP-2024-001',
      name: 'Ahmad Wijaya',
      email: 'ahmad@email.com',
      phone: '0812-3456-7890',
      region: 'Jakarta Selatan',
      category: 'Infrastruktur',
      title: 'Jalan Berlubang di Jl. Sudirman',
      content: 'Jalan di depan kantor sudah berlubang cukup parah dan membahayakan pengendara.',
      status: 'COMPLETED',
      response: 'Terima kasih atas laporannya. Jalan sudah diperbaiki pada tanggal 10 Januari 2024.',
      createdAt: '2024-01-05',
      updatedAt: '2024-01-10'
    },
    {
      id: 'LAP-2024-002',
      name: 'Siti Nurhaliza',
      email: 'siti@email.com',
      phone: '0813-9876-5432',
      region: 'Jakarta Pusat',
      category: 'Kesehatan',
      title: 'Puskesmas Kekurangan Obat',
      content: 'Puskesmas di daerah saya kekurangan obat-obatan dasar.',
      status: 'PROCESSING',
      response: null,
      createdAt: '2024-01-08',
      updatedAt: '2024-01-12'
    },
    {
      id: 'LAP-2024-003',
      name: 'Budi Santoso',
      email: 'budi@email.com',
      phone: '0815-2468-1357',
      region: 'Jakarta Barat',
      category: 'Pendidikan',
      title: 'Bantuan Sekolah Belum Cair',
      content: 'Bantuan sekolah untuk anak saya belum cair sampai sekarang.',
      status: 'RECEIVED',
      response: null,
      createdAt: '2024-01-12',
      updatedAt: '2024-01-12'
    },
    {
      id: 'LAP-2024-004',
      name: 'Dewi Lestari',
      email: 'dewi@email.com',
      phone: '0816-9753-2468',
      region: 'Jakarta Utara',
      category: 'Sosial',
      title: 'Bantuan Banjir',
      content: 'Membutuhkan bantuan korban banjir di wilayah saya.',
      status: 'ADDRESSED',
      response: 'Tim kami sudah mengunjungi lokasi dan menyalurkan bantuan awal.',
      createdAt: '2024-01-10',
      updatedAt: '2024-01-13'
    }
  ]

  const categories = [
    { value: 'all', label: 'Semua Kategori' },
    { value: 'INFRASTRUKTUR', label: 'Infrastruktur' },
    { value: 'PENDIDIKAN', label: 'Pendidikan' },
    { value: 'KESEHATAN', label: 'Kesehatan' },
    { value: 'EKONOMI', label: 'Ekonomi' },
    { value: 'SOSIAL', label: 'Sosial' },
    { value: 'KEAMANAN', label: 'Keamanan' },
    { value: 'LINGKUNGAN', label: 'Lingkungan' },
    { value: 'LAINNYA', label: 'Lainnya' }
  ]

  const regions = [
    { value: 'all', label: 'Semua Wilayah' },
    { value: 'Jakarta Pusat', label: 'Jakarta Pusat' },
    { value: 'Jakarta Utara', label: 'Jakarta Utara' },
    { value: 'Jakarta Barat', label: 'Jakarta Barat' },
    { value: 'Jakarta Selatan', label: 'Jakarta Selatan' },
    { value: 'Jakarta Timur', label: 'Jakarta Timur' }
  ]

  const statusConfig = {
    RECEIVED: {
      label: 'Diterima',
      color: 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300',
      icon: Clock,
      progress: 25
    },
    PROCESSING: {
      label: 'Diproses',
      color: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300',
      icon: AlertCircle,
      progress: 50
    },
    ADDRESSED: {
      label: 'Ditindaklanjuti',
      color: 'bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300',
      icon: CheckCircle,
      progress: 75
    },
    COMPLETED: {
      label: 'Selesai',
      color: 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300',
      icon: CheckCircle,
      progress: 100
    },
    REJECTED: {
      label: 'Ditolak',
      color: 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300',
      icon: XCircle,
      progress: 0
    }
  }

  const formatDate = (dateString: string) => {
    const date = new Date(dateString)
    return date.toLocaleDateString('id-ID', { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    })
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)
    
    // Simulasi pengiriman
    await new Promise(resolve => setTimeout(resolve, 2000))
    
    // Reset form
    setFormData({
      name: '',
      email: '',
      phone: '',
      region: '',
      category: '',
      title: '',
      content: '',
      attachment: null
    })
    
    setIsSubmitting(false)
    alert('Laporan berhasil dikirim! Kami akan segera menindaklanjuti.')
  }

  const handleInputChange = (field: string, value: string | File) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }))
  }

  const filteredReports = reports.filter(report => {
    const matchesSearch = report.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         report.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         report.id.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesStatus = selectedStatus === 'all' || report.status === selectedStatus
    const matchesCategory = selectedCategory === 'all' || report.category === selectedCategory
    
    return matchesSearch && matchesStatus && matchesCategory
  })

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Hero Section */}
      <section className="bg-gradient-to-br from-primary/10 to-secondary/10 py-16">
        <div className="container mx-auto px-4">
          <div className="text-center">
            <h1 className="text-4xl md:text-5xl font-bold mb-4">
              Laporan Masyarakat
            </h1>
            <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
              Sampaikan aspirasi, pengaduan, atau usulan Anda. Kami siap mendengar dan 
              menindaklanjuti setiap laporan dari masyarakat.
            </p>
          </div>
        </div>
      </section>

      {/* Statistics */}
      <section className="py-12 border-b">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            <Card className="text-center">
              <CardContent className="p-6">
                <div className="text-3xl font-bold text-primary mb-2">1,234</div>
                <div className="text-sm text-muted-foreground">Total Laporan</div>
              </CardContent>
            </Card>
            <Card className="text-center">
              <CardContent className="p-6">
                <div className="text-3xl font-bold text-green-600 mb-2">856</div>
                <div className="text-sm text-muted-foreground">Selesai</div>
              </CardContent>
            </Card>
            <Card className="text-center">
              <CardContent className="p-6">
                <div className="text-3xl font-bold text-yellow-600 mb-2">234</div>
                <div className="text-sm text-muted-foreground">Diproses</div>
              </CardContent>
            </Card>
            <Card className="text-center">
              <CardContent className="p-6">
                <div className="text-3xl font-bold text-blue-600 mb-2">144</div>
                <div className="text-sm text-muted-foreground">Diterima</div>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      <div className="container mx-auto px-4 py-12">
        <div className="grid lg:grid-cols-3 gap-8">
          {/* Form Section */}
          <div className="lg:col-span-1">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <MessageSquare className="mr-2 h-5 w-5" />
                  Buat Laporan Baru
                </CardTitle>
              </CardHeader>
              <CardContent>
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium mb-2">Nama Lengkap *</label>
                    <Input
                      value={formData.name}
                      onChange={(e) => handleInputChange('name', e.target.value)}
                      placeholder="Masukkan nama lengkap"
                      required
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">Email *</label>
                    <Input
                      type="email"
                      value={formData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      placeholder="email@example.com"
                      required
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">No. Telepon *</label>
                    <Input
                      value={formData.phone}
                      onChange={(e) => handleInputChange('phone', e.target.value)}
                      placeholder="08xx-xxxx-xxxx"
                      required
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">Wilayah *</label>
                    <Select value={formData.region} onValueChange={(value) => handleInputChange('region', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Pilih wilayah" />
                      </SelectTrigger>
                      <SelectContent>
                        {regions.map(region => (
                          <SelectItem key={region.value} value={region.value}>
                            {region.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">Kategori *</label>
                    <Select value={formData.category} onValueChange={(value) => handleInputChange('category', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Pilih kategori" />
                      </SelectTrigger>
                      <SelectContent>
                        {categories.map(category => (
                          <SelectItem key={category.value} value={category.value}>
                            {category.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">Judul Laporan *</label>
                    <Input
                      value={formData.title}
                      onChange={(e) => handleInputChange('title', e.target.value)}
                      placeholder="Judul singkat laporan"
                      required
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">Isi Laporan *</label>
                    <Textarea
                      value={formData.content}
                      onChange={(e) => handleInputChange('content', e.target.value)}
                      placeholder="Jelaskan secara detail laporan Anda"
                      rows={4}
                      required
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium mb-2">Lampiran (opsional)</label>
                    <div className="border-2 border-dashed border-muted-foreground/25 rounded-lg p-4 text-center">
                      <Upload className="h-8 w-8 mx-auto mb-2 text-muted-foreground" />
                      <p className="text-sm text-muted-foreground mb-2">
                        Upload foto, dokumen, atau file pendukung
                      </p>
                      <input
                        type="file"
                        onChange={(e) => handleInputChange('attachment', e.target.files?.[0] || null)}
                        className="hidden"
                        id="file-upload"
                      />
                      <Button type="button" variant="outline" size="sm" asChild>
                        <label htmlFor="file-upload" className="cursor-pointer">
                          Pilih File
                        </label>
                      </Button>
                    </div>
                  </div>

                  <Button 
                    type="submit" 
                    className="w-full" 
                    disabled={isSubmitting}
                  >
                    {isSubmitting ? (
                      'Mengirim...'
                    ) : (
                      <>
                        <Send className="mr-2 h-4 w-4" />
                        Kirim Laporan
                      </>
                    )}
                  </Button>
                </form>
              </CardContent>
            </Card>
          </div>

          {/* Reports List */}
          <div className="lg:col-span-2">
            <div className="space-y-6">
              {/* Filters */}
              <div className="flex flex-col sm:flex-row gap-4">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                  <Input
                    placeholder="Cari laporan..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
                
                <Select value={selectedStatus} onValueChange={setSelectedStatus}>
                  <SelectTrigger className="w-full sm:w-40">
                    <SelectValue placeholder="Status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Semua Status</SelectItem>
                    <SelectItem value="RECEIVED">Diterima</SelectItem>
                    <SelectItem value="PROCESSING">Diproses</SelectItem>
                    <SelectItem value="ADDRESSED">Ditindaklanjuti</SelectItem>
                    <SelectItem value="COMPLETED">Selesai</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger className="w-full sm:w-48">
                    <SelectValue placeholder="Kategori" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Semua Kategori</SelectItem>
                    <SelectItem value="Infrastruktur">Infrastruktur</SelectItem>
                    <SelectItem value="Pendidikan">Pendidikan</SelectItem>
                    <SelectItem value="Kesehatan">Kesehatan</SelectItem>
                    <SelectItem value="Sosial">Sosial</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {/* Reports List */}
              <div className="space-y-4">
                {filteredReports.map((report) => {
                  const statusInfo = statusConfig[report.status as keyof typeof statusConfig]
                  const StatusIcon = statusInfo.icon
                  
                  return (
                    <Card key={report.id} className="hover:shadow-md transition-shadow">
                      <CardContent className="p-6">
                        <div className="space-y-4">
                          {/* Header */}
                          <div className="flex items-start justify-between">
                            <div className="space-y-2">
                              <div className="flex items-center space-x-2">
                                <Badge variant="outline">{report.id}</Badge>
                                <Badge className={statusInfo.color}>
                                  <StatusIcon className="h-3 w-3 mr-1" />
                                  {statusInfo.label}
                                </Badge>
                                <Badge variant="secondary">{report.category}</Badge>
                              </div>
                              <h3 className="text-lg font-semibold">{report.title}</h3>
                            </div>
                          </div>

                          {/* Progress */}
                          <div className="space-y-2">
                            <div className="flex justify-between text-sm">
                              <span className="text-muted-foreground">Progress</span>
                              <span className="font-medium">{statusInfo.progress}%</span>
                            </div>
                            <Progress value={statusInfo.progress} className="h-2" />
                          </div>

                          {/* Content */}
                          <p className="text-muted-foreground">{report.content}</p>

                          {/* Reporter Info */}
                          <div className="grid grid-cols-1 sm:grid-cols-3 gap-4 text-sm">
                            <div className="flex items-center space-x-2">
                              <User className="h-4 w-4 text-muted-foreground" />
                              <span>{report.name}</span>
                            </div>
                            <div className="flex items-center space-x-2">
                              <MapPin className="h-4 w-4 text-muted-foreground" />
                              <span>{report.region}</span>
                            </div>
                            <div className="flex items-center space-x-2">
                              <Calendar className="h-4 w-4 text-muted-foreground" />
                              <span>{formatDate(report.createdAt)}</span>
                            </div>
                          </div>

                          {/* Response */}
                          {report.response && (
                            <div className="bg-muted/50 rounded-lg p-4">
                              <h4 className="font-medium mb-2">Tanggapan:</h4>
                              <p className="text-sm text-muted-foreground">{report.response}</p>
                            </div>
                          )}

                          {/* Actions */}
                          <div className="flex justify-between items-center pt-2">
                            <div className="text-sm text-muted-foreground">
                              Update: {formatDate(report.updatedAt)}
                            </div>
                            <Button variant="outline" size="sm">
                              <FileText className="h-4 w-4 mr-2" />
                              Detail
                            </Button>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  )
                })}
              </div>

              {filteredReports.length === 0 && (
                <div className="text-center py-12">
                  <MessageSquare className="h-12 w-12 mx-auto mb-4 text-muted-foreground" />
                  <h3 className="text-lg font-semibold mb-2">Tidak Ada Laporan</h3>
                  <p className="text-muted-foreground">
                    Tidak ada laporan yang ditemukan dengan filter yang dipilih.
                  </p>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
      
      <Footer />
    </div>
  )
}