'use client'

import { useState, useEffect } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { 
  Phone, 
  MessageCircle, 
  Clock, 
  Users, 
  Headphones,
  Mail,
  MapPin,
  Calendar,
  CheckCircle,
  X
} from 'lucide-react'

export default function CallCenter() {
  const [isOpen, setIsOpen] = useState(false)
  const [isOnline, setIsOnline] = useState(true)
  const [currentTime, setCurrentTime] = useState(new Date())

  useEffect(() => {
    const timer = setInterval(() => {
      setCurrentTime(new Date())
    }, 1000)

    return () => clearInterval(timer)
  }, [])

  const isOperatingHours = () => {
    const hour = currentTime.getHours()
    const day = currentTime.getDay()
    
    // Monday - Friday: 08:00 - 20:00
    // Saturday: 09:00 - 18:00
    // Sunday: 10:00 - 16:00
    if (day === 0) { // Sunday
      return hour >= 10 && hour < 16
    } else if (day === 6) { // Saturday
      return hour >= 9 && hour < 18
    } else { // Monday - Friday
      return hour >= 8 && hour < 20
    }
  }

  const getOperatingStatus = () => {
    if (!isOnline) return { status: 'offline', text: 'Offline', color: 'bg-gray-500' }
    if (isOperatingHours()) return { status: 'online', text: 'Online', color: 'bg-green-500' }
    return { status: 'closed', text: 'Closed', color: 'bg-red-500' }
  }

  const getOperatingHours = () => {
    const day = currentTime.getDay()
    if (day === 0) return '10:00 - 16:00'
    if (day === 6) return '09:00 - 18:00'
    return '08:00 - 20:00'
  }

  const handlePhoneCall = () => {
    window.open('tel:+6281234567890', '_blank')
  }

  const handleWhatsAppChat = () => {
    const message = encodeURIComponent('Halo, saya ingin berkonsultasi mengenai...')
    window.open(`https://wa.me/6281234567890?text=${message}`, '_blank')
  }

  const handleEmailContact = () => {
    window.open('mailto:info@politisi.id?subject=Pertanyaan dari Website', '_blank')
  }

  const operatingStatus = getOperatingStatus()

  if (!isOpen) {
    return (
      <div className="fixed bottom-4 left-4 z-40">
        <Button
          onClick={() => setIsOpen(true)}
          size="lg"
          className="rounded-full w-14 h-14 shadow-lg hover:shadow-xl transition-all duration-300 group bg-green-600 hover:bg-green-700"
        >
          <Headphones className="h-6 w-6 group-hover:scale-110 transition-transform" />
          <div className={`absolute top-0 right-0 w-4 h-4 ${operatingStatus.color} rounded-full border-2 border-background animate-pulse`} />
        </Button>
      </div>
    )
  }

  return (
    <div className="fixed bottom-4 left-4 z-40 w-80">
      <Card className="shadow-xl">
        {/* Header */}
        <CardHeader className="bg-gradient-to-r from-green-600 to-green-700 text-white p-4">
          <div className="flex items-center justify-between">
            <CardTitle className="text-lg flex items-center">
              <Headphones className="mr-2 h-5 w-5" />
              Call Center
            </CardTitle>
            <div className="flex items-center space-x-2">
              <Badge variant="secondary" className={`${operatingStatus.color} text-white border-none`}>
                {operatingStatus.text}
              </Badge>
              <Button
                variant="ghost"
                size="sm"
                onClick={() => setIsOpen(false)}
                className="text-white hover:bg-white/20"
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardHeader>

        <CardContent className="p-4 space-y-4">
          {/* Operating Hours */}
          <div className="bg-muted/50 rounded-lg p-3">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium">Jam Operasional</span>
              <Badge variant="outline" className="text-xs">
                Hari Ini: {getOperatingHours()}
              </Badge>
            </div>
            <div className="text-xs text-muted-foreground space-y-1">
              <div>Senin - Jumat: 08:00 - 20:00</div>
              <div>Sabtu: 09:00 - 18:00</div>
              <div>Minggu: 10:00 - 16:00</div>
            </div>
          </div>

          {/* Quick Actions */}
          <div className="space-y-3">
            <Button
              onClick={handlePhoneCall}
              className="w-full justify-start"
              disabled={!isOperatingHours()}
            >
              <Phone className="mr-2 h-4 w-4" />
              Telepon: +62 812-3456-7890
              {!isOperatingHours() && (
                <span className="ml-auto text-xs">(Tutup)</span>
              )}
            </Button>

            <Button
              onClick={handleWhatsAppChat}
              variant="outline"
              className="w-full justify-start"
            >
              <MessageCircle className="mr-2 h-4 w-4" />
              WhatsApp Chat
              <Badge variant="secondary" className="ml-auto text-xs">
                24/7
              </Badge>
            </Button>

            <Button
              onClick={handleEmailContact}
              variant="outline"
              className="w-full justify-start"
            >
              <Mail className="mr-2 h-4 w-4" />
              Email: info@politisi.id
            </Button>
          </div>

          {/* Services */}
          <div className="space-y-2">
            <h4 className="text-sm font-medium">Layanan Kami:</h4>
            <div className="space-y-2 text-xs text-muted-foreground">
              <div className="flex items-center">
                <CheckCircle className="h-3 w-3 mr-2 text-green-600" />
                Konsultasi Program
              </div>
              <div className="flex items-center">
                <CheckCircle className="h-3 w-3 mr-2 text-green-600" />
                Pengaduan Masyarakat
              </div>
              <div className="flex items-center">
                <CheckCircle className="h-3 w-3 mr-2 text-green-600" />
                Informasi Agenda
              </div>
              <div className="flex items-center">
                <CheckCircle className="h-3 w-3 mr-2 text-green-600" />
                Bantuan Darurat
              </div>
            </div>
          </div>

          {/* Statistics */}
          <div className="grid grid-cols-2 gap-3">
            <div className="text-center p-2 bg-muted/30 rounded-lg">
              <div className="text-lg font-bold text-green-600">24/7</div>
              <div className="text-xs text-muted-foreground">WhatsApp</div>
            </div>
            <div className="text-center p-2 bg-muted/30 rounded-lg">
              <div className="text-lg font-bold text-blue-600">
                {isOperatingHours() ? 'Online' : 'Offline'}
              </div>
              <div className="text-xs text-muted-foreground">Telepon</div>
            </div>
          </div>

          {/* Emergency Contact */}
          <div className="bg-red-50 border border-red-200 rounded-lg p-3">
            <div className="flex items-center space-x-2 mb-1">
              <Phone className="h-4 w-4 text-red-600" />
              <span className="text-sm font-medium text-red-800">Darurat</span>
            </div>
            <p className="text-xs text-red-700">
              Untuk keadaan darurat, hubungi: +62 811-222-333
            </p>
          </div>

          {/* Location */}
          <div className="flex items-center space-x-2 text-xs text-muted-foreground">
            <MapPin className="h-3 w-3" />
            <span>Jakarta, Indonesia</span>
          </div>
        </CardContent>
      </Card>
    </div>
  )
}