'use client'

import { useState, useEffect, useRef } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Badge } from '@/components/ui/badge'
import { ScrollArea } from '@/components/ui/scroll-area'
import { 
  MessageCircle, 
  Send, 
  X, 
  Minimize2, 
  Maximize2, 
  User,
  Bot,
  Circle
} from 'lucide-react'
import { io, Socket } from 'socket.io-client'

interface Message {
  id: string
  name: string
  email?: string
  message: string
  isAdmin: boolean
  timestamp: Date
  sessionId: string
}

export default function LiveChat() {
  const [isOpen, setIsOpen] = useState(false)
  const [isMinimized, setIsMinimized] = useState(false)
  const [messages, setMessages] = useState<Message[]>([])
  const [newMessage, setNewMessage] = useState('')
  const [userInfo, setUserInfo] = useState({ name: '', email: '' })
  const [isConnected, setIsConnected] = useState(false)
  const [isTyping, setIsTyping] = useState(false)
  const [showUserInfo, setShowUserInfo] = useState(true)
  const socketRef = useRef<Socket | null>(null)
  const messagesEndRef = useRef<HTMLDivElement>(null)
  const sessionIdRef = useRef(`session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`)

  useEffect(() => {
    if (isOpen && !socketRef.current) {
      // Initialize socket connection
      socketRef.current = io('/api/socketio')
      
      socketRef.current.on('connect', () => {
        setIsConnected(true)
        console.log('Connected to chat server')
      })
      
      socketRef.current.on('disconnect', () => {
        setIsConnected(false)
        console.log('Disconnected from chat server')
      })
      
      socketRef.current.on('message', (message: Message) => {
        setMessages(prev => [...prev, message])
      })
      
      socketRef.current.on('typing', (data: { isTyping: boolean; sessionId: string }) => {
        if (data.sessionId !== sessionIdRef.current) {
          setIsTyping(data.isTyping)
        }
      })
      
      // Add welcome message
      const welcomeMessage: Message = {
        id: `welcome_${Date.now()}`,
        name: 'Chatbot',
        message: 'Halo! Selamat datang di layanan chat kami. Silakan masukkan nama dan email Anda untuk memulai percakapan.',
        isAdmin: true,
        timestamp: new Date(),
        sessionId: 'bot'
      }
      setMessages([welcomeMessage])
    }
    
    return () => {
      if (socketRef.current) {
        socketRef.current.disconnect()
        socketRef.current = null
      }
    }
  }, [isOpen])

  useEffect(() => {
    scrollToBottom()
  }, [messages])

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
  }

  const handleSendMessage = () => {
    if (!newMessage.trim() || !socketRef.current) return
    
    const message: Message = {
      id: `msg_${Date.now()}`,
      name: userInfo.name,
      email: userInfo.email,
      message: newMessage,
      isAdmin: false,
      timestamp: new Date(),
      sessionId: sessionIdRef.current
    }
    
    socketRef.current.emit('message', message)
    setMessages(prev => [...prev, message])
    setNewMessage('')
    
    // Simulate bot response
    setTimeout(() => {
      const botResponse: Message = {
        id: `bot_${Date.now()}`,
        name: 'Chatbot',
        message: 'Terima kasih atas pesan Anda. Tim kami akan segera merespons. Mohon tunggu sebentar...',
        isAdmin: true,
        timestamp: new Date(),
        sessionId: 'bot'
      }
      setMessages(prev => [...prev, botResponse])
    }, 1000)
  }

  const handleUserInfoSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    if (userInfo.name.trim()) {
      setShowUserInfo(false)
      
      // Add user joined message
      const joinMessage: Message = {
        id: `join_${Date.now()}`,
        name: 'System',
        message: `${userInfo.name} telah bergabung dalam chat.`,
        isAdmin: true,
        timestamp: new Date(),
        sessionId: 'system'
      }
      setMessages(prev => [...prev, joinMessage])
    }
  }

  const formatTime = (date: Date) => {
    return date.toLocaleTimeString('id-ID', { 
      hour: '2-digit', 
      minute: '2-digit' 
    })
  }

  if (!isOpen) {
    return (
      <div className="fixed bottom-4 right-4 z-50">
        <Button
          onClick={() => setIsOpen(true)}
          size="lg"
          className="rounded-full w-14 h-14 shadow-lg hover:shadow-xl transition-all duration-300 group"
        >
          <MessageCircle className="h-6 w-6 group-hover:scale-110 transition-transform" />
          {isConnected && (
            <Circle className="absolute top-0 right-0 w-4 h-4 bg-green-500 rounded-full border-2 border-background" />
          )}
        </Button>
      </div>
    )
  }

  return (
    <div className="fixed bottom-4 right-4 z-50 w-96">
      <Card className="shadow-xl">
        {/* Header */}
        <CardHeader className="bg-primary text-primary-foreground p-4">
          <div className="flex items-center justify-between">
            <CardTitle className="text-lg flex items-center">
              <MessageCircle className="mr-2 h-5 w-5" />
              Live Chat
              <Badge variant="secondary" className="ml-2 bg-green-500 text-white">
                {isConnected ? 'Online' : 'Offline'}
              </Badge>
            </CardTitle>
            <div className="flex space-x-2">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => setIsMinimized(!isMinimized)}
                className="text-primary-foreground hover:bg-primary-foreground/20"
              >
                {isMinimized ? <Maximize2 className="h-4 w-4" /> : <Minimize2 className="h-4 w-4" />}
              </Button>
              <Button
                variant="ghost"
                size="sm"
                onClick={() => setIsOpen(false)}
                className="text-primary-foreground hover:bg-primary-foreground/20"
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardHeader>

        {!isMinimized && (
          <>
            {/* Messages */}
            <CardContent className="p-0">
              <ScrollArea className="h-96 p-4">
                <div className="space-y-4">
                  {messages.map((message) => (
                    <div
                      key={message.id}
                      className={`flex ${message.isAdmin ? 'justify-start' : 'justify-end'}`}
                    >
                      <div
                        className={`max-w-xs px-4 py-2 rounded-lg ${
                          message.isAdmin
                            ? 'bg-muted text-muted-foreground'
                            : 'bg-primary text-primary-foreground'
                        }`}
                      >
                        <div className="flex items-center space-x-2 mb-1">
                          {message.isAdmin ? (
                            <Bot className="h-3 w-3" />
                          ) : (
                            <User className="h-3 w-3" />
                          )}
                          <span className="text-xs font-medium">
                            {message.name}
                          </span>
                          <span className="text-xs opacity-70">
                            {formatTime(message.timestamp)}
                          </span>
                        </div>
                        <p className="text-sm">{message.message}</p>
                      </div>
                    </div>
                  ))}
                  
                  {isTyping && (
                    <div className="flex justify-start">
                      <div className="bg-muted text-muted-foreground px-4 py-2 rounded-lg">
                        <div className="flex space-x-1">
                          <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" />
                          <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.1s' }} />
                          <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.2s' }} />
                        </div>
                      </div>
                    </div>
                  )}
                  
                  <div ref={messagesEndRef} />
                </div>
              </ScrollArea>
            </CardContent>

            {/* User Info Form */}
            {showUserInfo && (
              <CardContent className="p-4 border-t">
                <form onSubmit={handleUserInfoSubmit} className="space-y-3">
                  <Input
                    placeholder="Nama Anda"
                    value={userInfo.name}
                    onChange={(e) => setUserInfo(prev => ({ ...prev, name: e.target.value }))}
                    required
                  />
                  <Input
                    type="email"
                    placeholder="Email (opsional)"
                    value={userInfo.email}
                    onChange={(e) => setUserInfo(prev => ({ ...prev, email: e.target.value }))}
                  />
                  <Button type="submit" className="w-full">
                    Mulai Chat
                  </Button>
                </form>
              </CardContent>
            )}

            {/* Message Input */}
            {!showUserInfo && (
              <CardContent className="p-4 border-t">
                <div className="flex space-x-2">
                  <Input
                    placeholder="Ketik pesan..."
                    value={newMessage}
                    onChange={(e) => setNewMessage(e.target.value)}
                    onKeyPress={(e) => e.key === 'Enter' && handleSendMessage()}
                    disabled={!isConnected}
                  />
                  <Button
                    onClick={handleSendMessage}
                    disabled={!newMessage.trim() || !isConnected}
                    size="sm"
                  >
                    <Send className="h-4 w-4" />
                  </Button>
                </div>
                {!isConnected && (
                  <p className="text-xs text-muted-foreground mt-2">
                    Terputus dari server. Mencoba menghubungkan kembali...
                  </p>
                )}
              </CardContent>
            )}
          </>
        )}
      </Card>
    </div>
  )
}