import { Card, CardContent } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Calendar, MapPin, Clock, ArrowRight } from 'lucide-react'
import Link from 'next/link'

export default function AgendaKegiatan() {
  // Sample data - akan diganti dengan data dari database
  const agenda = [
    {
      id: 1,
      title: "Reses Masyarakat di Jakarta Selatan",
      description: "Menjaring aspirasi masyarakat dan menyerap keluhan langsung dari warga",
      date: "2024-01-15",
      time: "09:00 - 12:00",
      location: "Kantor Kecamatan Kebayoran Lama",
      type: "Reses",
      status: "upcoming"
    },
    {
      id: 2,
      title: "Rapat Kerja Komisi X DPR RI",
      description: "Pembahasan anggaran pendidikan tahun 2024",
      date: "2024-01-18",
      time: "13:00 - 16:00",
      location: "Gedung DPR RI, Jakarta",
      type: "Rapat",
      status: "upcoming"
    },
    {
      id: 3,
      title: "Sosialisasi Program Kesehatan Gratis",
      description: "Mengenalkan program kesehatan gratis kepada masyarakat",
      date: "2024-01-20",
      time: "10:00 - 14:00",
      location: "Puskesmas Kecamatan Mampang",
      type: "Sosialisasi",
      status: "upcoming"
    }
  ]

  const typeColors: Record<string, string> = {
    "Reses": "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300",
    "Rapat": "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300",
    "Sosialisasi": "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-300",
    "Kunjungan": "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-300",
    "Kampaye": "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300"
  }

  const formatDate = (dateString: string) => {
    const date = new Date(dateString)
    const options: Intl.DateTimeFormatOptions = { 
      weekday: 'long', 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    }
    return date.toLocaleDateString('id-ID', options)
  }

  const isToday = (dateString: string) => {
    const today = new Date()
    const date = new Date(dateString)
    return date.toDateString() === today.toDateString()
  }

  const isPast = (dateString: string) => {
    const today = new Date()
    const date = new Date(dateString)
    return date < today
  }

  return (
    <section className="py-20 bg-muted/30">
      <div className="container mx-auto px-4">
        <div className="flex justify-between items-center mb-12">
          <div>
            <h2 className="text-3xl md:text-4xl font-bold mb-4">
              Agenda & Kegiatan
            </h2>
            <p className="text-lg text-muted-foreground">
              Jadwal kegiatan dan program kerja terkini
            </p>
          </div>
          <Link href="/agenda">
            <Button variant="outline" className="group">
              Lihat Semua
              <ArrowRight className="ml-2 h-4 w-4 transition-transform group-hover:translate-x-1" />
            </Button>
          </Link>
        </div>

        <div className="grid lg:grid-cols-3 gap-6">
          {agenda.map((item) => {
            const today = isToday(item.date)
            const past = isPast(item.date)
            
            return (
              <Card key={item.id} className={`hover:shadow-lg transition-all duration-300 ${today ? 'ring-2 ring-primary' : ''} ${past ? 'opacity-75' : ''}`}>
                <CardContent className="p-6">
                  <div className="space-y-4">
                    {/* Header */}
                    <div className="flex items-start justify-between">
                      <div className="space-y-2">
                        <div className="flex items-center space-x-2">
                          <Badge 
                            variant="secondary" 
                            className={typeColors[item.type] || 'bg-gray-100 text-gray-800'}
                          >
                            {item.type}
                          </Badge>
                          {today && (
                            <Badge className="bg-primary text-primary-foreground">
                              Hari Ini
                            </Badge>
                          )}
                          {past && (
                            <Badge variant="outline">
                              Selesai
                            </Badge>
                          )}
                        </div>
                        <h3 className="text-lg font-semibold line-clamp-2">
                          {item.title}
                        </h3>
                      </div>
                    </div>

                    {/* Description */}
                    <p className="text-muted-foreground text-sm line-clamp-2">
                      {item.description}
                    </p>

                    {/* Details */}
                    <div className="space-y-2">
                      <div className="flex items-center text-sm text-muted-foreground">
                        <Calendar className="h-4 w-4 mr-2 flex-shrink-0" />
                        <span>{formatDate(item.date)}</span>
                      </div>
                      <div className="flex items-center text-sm text-muted-foreground">
                        <Clock className="h-4 w-4 mr-2 flex-shrink-0" />
                        <span>{item.time}</span>
                      </div>
                      <div className="flex items-center text-sm text-muted-foreground">
                        <MapPin className="h-4 w-4 mr-2 flex-shrink-0" />
                        <span className="line-clamp-1">{item.location}</span>
                      </div>
                    </div>

                    {/* Action */}
                    <div className="pt-2">
                      {!past && (
                        <Button size="sm" className="w-full">
                          {today ? 'Gabung Sekarang' : 'Ingatkan Saya'}
                        </Button>
                      )}
                      {past && (
                        <Button size="sm" variant="outline" className="w-full">
                          Lihat Dokumentasi
                        </Button>
                      )}
                    </div>
                  </div>
                </CardContent>
              </Card>
            )
          })}
        </div>

        {/* Calendar View Link */}
        <div className="mt-12 text-center">
          <Card className="max-w-2xl mx-auto p-8">
            <div className="space-y-4">
              <h3 className="text-xl font-semibold">
                Lihat Kalender Lengkap
              </h3>
              <p className="text-muted-foreground">
                Akses kalender lengkap untuk melihat semua jadwal kegiatan dan bisa disinkronkan dengan Google Calendar
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button variant="outline">
                  📅 Google Calendar
                </Button>
                <Button variant="outline">
                  📱 Download iCal
                </Button>
              </div>
            </div>
          </Card>
        </div>
      </div>
    </section>
  )
}