import { Metadata } from 'next'

interface SEOProps {
  title?: string
  description?: string
  keywords?: string[]
  image?: string
  url?: string
  type?: 'website' | 'article'
  publishedTime?: string
  modifiedTime?: string
  author?: string
  section?: string
  tags?: string[]
}

export function generateSEO({
  title = 'Politisi Indonesia - Melayani Rakyat',
  description = 'Website resmi politisi Indonesia. Berjuang untuk kemakmuran dan keadilan sosial bagi seluruh rakyat Indonesia.',
  keywords = ['politisi', 'Indonesia', 'pelayanan', 'rakyat', 'visi misi', 'program kerja'],
  image = '/images/og-image.jpg',
  url = 'https://politisi.id',
  type = 'website',
  publishedTime,
  modifiedTime,
  author,
  section,
  tags
}: SEOProps): Metadata {
  const siteName = 'Politisi Indonesia'
  const titleTemplate = title === siteName ? title : `${title} | ${siteName}`
  
  return {
    title: titleTemplate,
    description,
    keywords: keywords.join(', '),
    authors: [{ name: author || 'Tim Media Politisi Indonesia' }],
    creator: author || 'Tim Media Politisi Indonesia',
    publisher: 'Politisi Indonesia',
    robots: {
      index: true,
      follow: true,
      googleBot: {
        index: true,
        follow: true,
        'max-video-preview': -1,
        'max-image-preview': 'large',
        'max-snippet': -1,
      },
    },
    openGraph: {
      type,
      locale: 'id_ID',
      url,
      title: titleTemplate,
      description,
      siteName,
      images: [
        {
          url: image,
          width: 1200,
          height: 630,
          alt: title,
        },
      ],
      publishedTime,
      modifiedTime,
      section,
      tags,
    },
    twitter: {
      card: 'summary_large_image',
      title: titleTemplate,
      description,
      images: [image],
      creator: '@politisi_id',
      site: '@politisi_id',
    },
    alternates: {
      canonical: url,
      languages: {
        'id-ID': url,
        'en-US': `${url}/en`,
      },
    },
    verification: {
      google: 'your-google-verification-code',
      yandex: 'your-yandex-verification-code',
    },
    other: {
      'article:author': author,
      'article:section': section,
      'article:tag': tags?.join(','),
      'og:site_name': siteName,
      'twitter:domain': 'politisi.id',
    },
  }
}

export function generateArticleSEO({
  title,
  description,
  keywords,
  image,
  url,
  publishedTime,
  modifiedTime,
  author,
  section,
  tags
}: SEOProps): Metadata {
  return generateSEO({
    title,
    description,
    keywords,
    image,
    url,
    type: 'article',
    publishedTime,
    modifiedTime,
    author,
    section,
    tags
  })
}

export function generateJsonLd({
  type,
  name,
  description,
  url,
  image,
  datePublished,
  dateModified,
  author,
  publisher,
  mainEntityOfPage,
  breadcrumbList,
  article,
  organization,
  website
}: any) {
  const jsonLd = {
    '@context': 'https://schema.org',
    '@type': type,
    name,
    description,
    url,
    image,
    datePublished,
    dateModified,
    author,
    publisher,
    mainEntityOfPage,
    breadcrumbList,
    article,
    organization,
    website
  }

  // Remove undefined values
  Object.keys(jsonLd).forEach(key => {
    if (jsonLd[key] === undefined) {
      delete jsonLd[key]
    }
  })

  return jsonLd
}

export function generateWebsiteJsonLd() {
  return generateJsonLd({
    type: 'WebSite',
    name: 'Politisi Indonesia',
    description: 'Website resmi politisi Indonesia. Berjuang untuk kemakmuran dan keadilan sosial bagi seluruh rakyat Indonesia.',
    url: 'https://politisi.id',
    image: '/images/og-image.jpg',
    organization: {
      '@type': 'Organization',
      name: 'Politisi Indonesia',
      url: 'https://politisi.id',
      logo: {
        '@type': 'ImageObject',
        url: '/images/logo.png',
        width: 512,
        height: 512,
      },
      contactPoint: {
        '@type': 'ContactPoint',
        telephone: '+62-812-3456-7890',
        contactType: 'customer service',
        availableLanguage: ['Indonesian', 'English'],
      },
      sameAs: [
        'https://facebook.com/politisiindonesia',
        'https://twitter.com/politisiid',
        'https://instagram.com/politisiindonesia',
        'https://youtube.com/c/politisiindonesia',
      ],
    },
    website: {
      '@type': 'WebSite',
      name: 'Politisi Indonesia',
      url: 'https://politisi.id',
      potentialAction: {
        '@type': 'SearchAction',
        target: {
          '@type': 'EntryPoint',
          urlTemplate: 'https://politisi.id/search?q={search_term_string}',
        },
        'query-input': 'required name=search_term_string',
      },
    },
  })
}

export function generateArticleJsonLd({
  title,
  description,
  url,
  image,
  datePublished,
  dateModified,
  author,
  category
}: any) {
  return generateJsonLd({
    type: 'NewsArticle',
    name: title,
    description,
    url,
    image,
    datePublished,
    dateModified,
    author: {
      '@type': 'Person',
      name: author,
    },
    publisher: {
      '@type': 'Organization',
      name: 'Politisi Indonesia',
      logo: {
        '@type': 'ImageObject',
        url: '/images/logo.png',
        width: 512,
        height: 512,
      },
    },
    mainEntityOfPage: {
      '@type': 'WebPage',
      '@id': url,
    },
    article: {
      '@type': 'Article',
      headline: title,
      description,
      image,
      datePublished,
      dateModified,
      author: {
        '@type': 'Person',
        name: author,
      },
      publisher: {
        '@type': 'Organization',
        name: 'Politisi Indonesia',
        logo: {
          '@type': 'ImageObject',
          url: '/images/logo.png',
          width: 512,
          height: 512,
        },
      },
    },
  })
}