import { Server } from 'socket.io';

interface Message {
  id: string;
  name: string;
  email?: string;
  message: string;
  isAdmin: boolean;
  timestamp: Date;
  sessionId: string;
}

export const setupSocket = (io: Server) => {
  io.on('connection', (socket) => {
    console.log('Client connected:', socket.id);
    
    // Handle chat messages
    socket.on('message', (msg: Message) => {
      console.log('Message received:', msg);
      
      // Broadcast message to all connected clients
      io.emit('message', msg);
      
      // Here you can also save message to database
      // saveMessageToDatabase(msg);
    });

    // Handle typing indicators
    socket.on('typing', (data: { isTyping: boolean; sessionId: string }) => {
      // Broadcast typing status to other clients
      socket.broadcast.emit('typing', data);
    });

    // Handle join room (for admin chat)
    socket.on('join-room', (roomId: string) => {
      socket.join(roomId);
      console.log(`Client ${socket.id} joined room ${roomId}`);
    });

    // Handle admin messages
    socket.on('admin-message', (data: { roomId: string; message: Message }) => {
      // Send message to specific room
      io.to(data.roomId).emit('message', data.message);
    });

    // Handle disconnect
    socket.on('disconnect', () => {
      console.log('Client disconnected:', socket.id);
    });

    // Send welcome message
    socket.emit('message', {
      id: `welcome_${Date.now()}`,
      name: 'System',
      message: 'Welcome to Live Chat Support!',
      isAdmin: true,
      timestamp: new Date(),
      sessionId: 'system'
    } as Message);
  });
};